#pragma once
#include <stdafx.h>
#ifdef PYBIND11_DEF
#include <Python.h>
#include <pybind11/pybind11.h>
#include <pybind11/numpy.h>
#endif


/**
 * \brief This is the main header to export the 3DCoat API to Python. Please follow strict rules for documenting each function you insert there.
 * 1. Each function should have a brief description, detailed description is optional.
 * 2. If function has parameters, each parameter should have a description.
 * 3. If function returns something, it should have a description.
 * 4. It is better to insert some examples of usage if possible.
 * 5. Avoid nested classes whenever possible (this is not mandatory).
 * 6. Always use std::string or const char* for strings, std::vector for lists.
 * 7. If you introduce some function with callbacks, duplicate it with the pybind11::object, see the ui::cmd as an example.
 * 5. Add APICALL macro to each class declaration, other vice it will not appear in the list of exported functions.
 */
class ItemsTree;
class ItemsFolder;
class ClusteredMesh;

#ifndef PY_PARSER
#ifdef PYBIND11_DEF

#define PY_BYTE_ARRAY pybind11::array_t<unsigned char, pybind11::array::c_style>
#define PY_WORD_ARRAY pybind11::array_t<WORD, pybind11::array::c_style>
#define PY_DWORD_ARRAY pybind11::array_t<dword, pybind11::array::c_style>
#define PY_INT_ARRAY pybind11::array_t<int, pybind11::array::c_style>
#define PY_FLOAT_ARRAY pybind11::array_t<float, pybind11::array::c_style>
#define PY_DOUBLE_ARRAY pybind11::array_t<double, pybind11::array::c_style>

#endif
#endif

#pragma pack(push, 1)
/**
 * \brief The coat namespace used for most 3DCoat API calls except low-level internal structures.
 */

namespace coat {
	class Model;
	class Volume;
	/// 4D - float vector, see the cVec4
	typedef comms::cVec4 vec4;

	/// 3D - float vector, see the cVec3
	typedef comms::cVec3 vec3;

	/// 3D - int vector, see the cVec3i
	typedef comms::cVec3i vec3i;

	/// 2D - vector, see the cVec2
	typedef comms::cVec2 vec2;

	/// 4x4 float matrix, see the cMat4
	typedef comms::cMat4 mat4;

	/// 3x3 float matrix, see the cMat3
	typedef comms::cMat3 mat3;

	typedef comms::cMath math;

	/// the string that is compatible with the 3DCoat engine, see the cStr
	typedef comms::cStr str;

	/// the array template, see cList
	template <class X> using list = comms::cList<X>;

	/// rect float, see the cRect 
	typedef comms::cRect rect;

	typedef comms::cQuat quat;

	typedef comms::cRotation rotation;

	typedef comms::cAngles angles;

	// axis-aligned bound box
	typedef comms::cBounds boundbox;

	/**
	 * \brief Boolean operations type
	 */
	enum  BoolOpType {
		/// just merge, no booleans, it may be used only in surface mode
		BOOL_MERGE = -1,
		/// boolean add
		BOOL_ADD = 0,
		/// boolean subtract
		BOOL_SUBTRACT = 1,
		/// boolean intersect
		BOOL_INTERSECT = 2,
	};
	class prim;


	/**
	 * \brief strat the main menu root items, this command may be called only from the menu making script
	 * \param id the menu id, see examples in the `UserPrefs/Rooms/menu.py` or `cpp`
	 */
	void start_main_menu(const char* id);

	/**
	 * \brief add the item to the menu, this command may be called only from the menu making script
	 * \param id the item id, see examples in the `UserPrefs/Rooms/menu.py` or `cpp`. If you want to trigger some script by this menu item
	 * you may use '$execute:path/to/your/script.py' as the id.
	 */
	void menu_item(const char* id);

	/**
	 * \brief add the information item to the menu (without any action, just message), this command may be called only from the menu making script
	 * \param id the item text identifier
	 */
	void menu_info(const char* id);
	
	/**
	 * \brief add the submenu to the current menu, this command may be called only from the menu making script
	 * \param id the submenu id/text
	 * \return always True, just for structuring the script
	 */
	bool menu_submenu(const char* id);

	/**
	 * \brief finish the current submenu, this command may be called only from the menu making script
	 */
	void menu_exit();

	/**
	 * \brief add the separator to the current menu, this command may be called only from the menu making script
	 */
	void menu_separator();

	/**
	 * \brief set the hotkey for the menu item, this command may be called only from the menu making script
	 * \param id the item id, see examples in the `UserPrefs/Rooms/menu.py` or `cpp`
	 * \param Shift set true if Shift should be pressed
	 * \param Ctrl set true if Ctrl should be pressed
	 * \param Alt set true if Alt should be pressed
	 */
	void menu_hotkey(const char* id, int Shift, int Ctrl, int Alt);

	/**
	 * \brief start the menu that will be shown on the icon click, like the navigation menu
	 * \param id the text identifier 
	 * \param size the size of the icon
	 * \return always True to structure the script
	 */
	bool iconic_submenu(const char* id, int size);
	
	/**
	 * \brief is the scene new/empty?
	 * \return True if empty
	 */
	bool is_new_scene();
	
	/**
	 * \brief is it steam app?
	 * \return True if it is
	 */
	bool is_steam_app();
	
	/**
	 * \brief check if the app is medical
	 * \return True if it is
	 */
	bool is_medical();
	
	/**
	 * \brief chack if there are any ppp objects in scene
	 * \return True if there are
	 */
	bool is_ppp();

	/**
	 * \brief check if current sculpt object is in proxy mode
	 * \return True if it is
	 */
	bool is_proxy();

	/**
	 * \brief check if the current sculpt object is on some multiresolution level
	 * \return True if it is
	 */
	bool is_multires();

	/**
	 * \brief check if the current sculpt object is in surface mode
	 * \return True if it is
	 */
	bool is_surface();

	/**
	 * \brief check if you are in some room
	 * \param name the room name
	 * \return True if you are in that room
	 */
	bool IsInRoom(const char* name);

	/**
	 * \brief check if the room exists
	 * \param name the room name/identifier
	 * \return True if the room exists
	 */
	bool RoomExists(const char* name);

	/**
	 * \brief check if the file exists
	 * \param path the file path, full or relative to Coat's documents
	 * \return True if exists
	 */
	bool CheckIfExists(const char* path);

	
	/**
	 * \brief check is scripts recording available
	 * \return True if available
	 */
	bool UseRecordScript();


	/**
	 * \brief check if mv objects available in scene
	 * \return True if available
	 */
	bool is_mv();

	/**
	 * \brief check if ptex is used in the current scene
	 * \return True if used
	 */
	bool is_ptex();

	/**
	 * \brief show the rmb panel
	 */
	void show_rmb_panel();

	/**
	 * \brief show the space panel (with limitations if need)
	 * \param Subset the subset, if need
	 * \param NumColumns amount of columns
	 */
	void show_space_panel(const char* Subset, int NumColumns);

	/**
	 * \brief gltf export supported
	 * \return True if supported
	 */
	bool gltf_support();

	/**
	 * \brief returns the texturing approach index (from the Textures menu)
	 * \return the index
	 */
	int tex_approach();

	/**
	 * \brief insert extension into the main menu (may be called only from the menu making script)
	 * \param id the extension id
	 */
	void menu_insert_extensions(const char* id);

	/**
	 * \brief insert extension menu into the main menu
	 */
	void extensions_main_menu();

	/**
	 * \brief insert extensions into the toolset (may be used only from the toolset.py)
	 */
	void insert_extensions();

	/**
	 * \brief set the space panel columns count (only for the toolset.py)
	 * \param num amount of columns
	 */
	void set_space_panel_columns_count(int num);

	/**
	 * \brief set the default value for auto-snapping, usually for the retopo/modeling rooms (in toolset.py)
	 * \param value the default value
	 */
	void SetAutoSnapDefaults(bool value);

	/**
	 * \brief returns boolean property value
	 * \param id the property id, available values: RMBObjectInCache, RMBObjectIsSurface, ObjectHasNodes, IsCurvePrimitive, IsCurveClosed, OverSculptObject
	 * \return the property value
	 */
	bool menu_property(const char* id);

	/**
	 * \brief start the tools section in the toolset.py
	 * \param id the section id
	 */
	void tools_section(const char* id);

	/**
	 * \brief add the item to the tools section (toolset.py)
	 * \param id the tool identifier
	 */
	void tools_item(const char* id);

	/**
	 * \brief set the additional suffix for the page in the toolset.py
	 * \param suffix usually "S" or "V" 
	 */
	void page_suffix(const char* suffix);

	/**
	 * \brief set the default tool for the toolset.py
	 * \param tool the tool identifier
	 */
	void default_tool(const char* tool);

	/**
	 * \brief is Debug mode (for developers only)
	 * \return True if debug
	 */
	bool IsDebug();

	/**
	 * \brief start the RNB panel. This command may be called only from the RMB response making script (curves.py, rmb.py)
	 */
	void start_rmb_panel();

	/**
	 * \brief sort items in the current menu
	 */
	void menu_sort();

	/**
	 * \brief is the script recording enabled?
	 * \return True if enabled
	 */
	bool IsRecordScript();

	/**
	 * \brief check if we are in some tool
	 * \param ToolID the tool identifier
	 * \return True if in that tool
	 */
	bool IsInTool(const char* ToolID);

	/**
	 * \brief check if the VoxTree item is picked
	 * \return true if picked
	 */
	bool voxtree_item_picked();

	/**
	 * \brief the retopo object is picked
	 * \return True if picked
	 */
	bool retopo_object_picked();

	/**
	 * \brief check if no object is picked
	 * \return True if no object is picked
	 */
	bool empty_space_picked();

	/**
	 * \brief check if the sculpt object is picked in the viewport
	 * \return True if picked
	 */
	bool voxtree_object_picked();

	/**
	 * \brief get the current tool submode (usually for the uv/retopo tools)
	 * \param id the submode identifier
	 * \return the value
	 */
	int GetCurrentToolSubmode(const char* id);

	/**
	 * \brief comment in toolset.py for auto-documentation (legacy)
	 * \param id the text of the comment
	 */
	void tools_comment(const char* id);

	/**
	 * \brief check if script is in auto-documenting mode (legacy)
	 * \return the state
	 */
	bool doc_mode();

	/**
	 * \brief enable the radial menu mode for the space panel
	 */
	void PureIconic();

	/**
	 * \brief check if UI changes locked (for specialized applications, like printing)
	 * \return the lock state
	 */
	bool lock_ui_changes();


	/**
	 * \brief returns if ue5 support enabled
	 * \return True if enabled
	 */
	bool ue5_support();

	/**
	 * \brief run extension
	 * \param name of the extension
	 */
	void run_extension(const char* extension_name, bool auto_start = false);

	/**
	 * \brief The mesh reference
	 */
	class APICALL Mesh {
	protected:
		friend class Model;
		static std::vector<std::pair<comms::cMeshContainer*, int>> allocated_meshes;
		comms::cMeshContainer* allocate();
		void inc_ref(comms::cMeshContainer* m);
		void dec_ref(comms::cMeshContainer* m);

		comms::cMeshContainer* MeshData;
		list<int> raw_start;
		int raw_count;
		void validate_raws();
	public:
		Mesh();
		~Mesh();

		Mesh(const Mesh& m);
		Mesh(const Model& m);
		Mesh& operator = (const Mesh& m);
		Mesh& operator = (const Model& m);
		Mesh MakeCopy();

		/**
		 * \brief Load the mesh from the file.
		 * \param name the filename. May contain full path or relative to the coat's install or documents folder.
		 * \return true if successful.
		 */
		bool Read(const char* name);

		/**
		 * \brief Save the mesh to file
		 * \param name Full or relative path
		 * \return true if successful
		 */
		bool Write(const char* name);

		/**
		 * \brief Check if mesh is valid
		 * \return true if mesh is valid
		 */
		bool valid() const;

		/**
		 * \brief The low-level mesh reference allows to create, operate over individual faces, vertices, objects
		 * \return returns the low-level mesh reference
		 */
		comms::cMeshContainer* geometry();

		comms::cMeshContainer* geometry() const;

		/**
		 * \brief clear the mesh
		 */
		void clear();

		/**
		 * \brief concatenate with the other mesh
		 * \param m the mesh to be concatenated with this one
		 */
		Mesh& operator +=(const Mesh& m);

		/**
		 * \brief concatenate with the Model
		 * \param m the mesh to be concatenated with this one
		 */
		Mesh& operator +=(const Model& m);

		/**
		 * \brief concatenate the transformed mesh with the current one
		 * \param m the mesh
		 * \param t the transform
		 */
		void addTransformed(const Mesh& m, const mat4& t);

		/**
		 * \brief boolean operation
		 * \param m the mesh to operate
		 */
		void boolean(const Mesh& m, BoolOpType op);

		/**
		 * \brief transform the mesh
		 * \param transform the transformation matrix
		 */
		void transform(const mat4& transform);

		/**
		 * \brief rotate the mesh so that X axis will be aligned with axisX, Y axis will be aligned with axisY
		 * \param axisX the new X axis
		 * \param axisY the new Y axis
		 */
		void rotateToXYAxis(const vec3& axisX, const vec3& axisY);

		/**
		 * \brief rotate the mesh so that Y axis will be aligned with axisY, Z axis will be aligned with axisZ
		 * \param axisY the new Y axis
		 * \param axisZ the new Z axis
		 */
		void rotateToYZAxis(const vec3& axisY, const vec3& axisZ);

		/**
		 * \brief rotate the mesh so that Z axis will be aligned with axisZ, X axis will be aligned with axisX
		 * \param axisZ the new Z axis
		 * \param axisX the new X axis
		 */
		void rotateToZXAxis(const vec3& axisZ, const vec3& axisX);

		/**
		 * \brief returns the amount of verts in the mesh
		 * \return the amount
		 */
		int vertsCount() const;

		/**
		 * \brief returns the amount of UV - verts in the mesh
		 * \return the amount
		 */
		int vertsUvCount() const;

		/**
		 * \brief returns the amount of normal - verts in the mesh
		 * \return teh amount
		 */
		int vertsNormalCount() const;

		/**
		 * \brief returns the faces amount
		 * \return the amount
		 */
		int facesCount();

		/**
		 * \brief get the vertex coordinate
		 * \param idx the vertex index
		 * \return the coordinate
		 */
		vec3 getVertex(int idx) const;

		/**
		 * \brief set the vertex coordinate
		 * \param idx the vertex index
		 * \param v the coordinate
		 */
		void setVertex(int idx, const vec3& v);

		/**
		 * \brief create the positional vertex
		 * \param position the position
		 * \return the positional vertex index
		 */
		int createNewVertex(const vec3& position);

		/**
		 * \brief get the UV coordinate of the vertex, pay attention position verts and UV verts are different, they have different indices
		 * \param idx the UV vertex index, [0..vertsUvCount() - 1]
		 * \return the UV coordinate
		 */
		vec2 getVertexUV(int idx) const;

		/**
		 * \brief set the UV coordinate of the vertex, pay attention position verts and UV verts are different, they have different indices
		 * \param idx the UV vrertex index, [0..vertsUvCount() - 1]
		 * \param v the UV coordinate
		 */
		void setVertexUV(int idx, const vec2& v);

		/**
		 * \brief create new UV vertex to be used for faces
		 * \param uv the texture coordinates
		 * \return the index
		 */
		int createNewUvVertex(const vec2& uv);

		/**
		 * \brief get the normal of the vertex, pay attention position verts and normal verts are different, they have different indices
		 * \param idx the normal vertex index, [0..vertsNormalCount() - 1]
		 * \return the normal
		 */
		vec3 getVertexNormal(int idx) const ;

		/**
		 * \brief set the normal of the vertex, pay attention position verts and normal verts are different, they have different indices
		 * \param idx the normal vertex index, [0..vertsNormalCount() - 1]
		 * \param v the normal
		 */
		void setVertexNormal(int idx, const vec3& v);

		/**
		 * \brief re-calculate normals over the mesh
		 */
		void calcNormals();

		/**
		 * \brief re-calculate normals over the mesh, ignore the sharp edges
		 */
		void calcNormalsIgnoreSharpEdges();

		/**
		 * \brief get the amount of vertices over the face
		 * \param face the face index, should be in [0..facesCount() - 1]
		 * \return the verts amount
		 */
		int getFaceVertsCount(int face);

		/**
		 * \brief get the amount of UV vertices over the face
		 * \param face the face index
		 * \return amount of vertices over the face, 0 if UV-s not assigned
		 */
		int getFaceUvVertsCount(int face);

		/**
		 * \brief get the positional vertex index over the face
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param faceVertexIndex the index of the vertex within the face, should be in [0..getFaceVertsCount(faceIndex) - 1]
		 * \return the positional vertex index
		 */
		int getFaceVertex(int faceIndex, int faceVertexIndex);

		/**
		 * \brief get the list of UV vertex indices over the face, pay attention UV vertices are not same as position vertices
		 * \param face the face index
		 * \return the list of vertex indices
		 */
		std::vector<int> getFaceVerts(int face);
		void getFaceVerts(int face, list<int>& vertices);

		/**
		 * \brief set the list of positional vertex indices over the face
		 * \param face the face index
		 * \param vertices the list of vertex indices
		 */
		void setFaceVerts(int face, const std::vector<int>& vertices);
		void setFaceVerts(int face, const list<int>& vertices);

		/**
		 * \brief get the UV vertex index over the face
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param faceVertexIndex the index of the vertex within the face, should be in [0..getFaceVertsCount(faceIndex) - 1]
		 * \return the UV vertex index, -1 if no UVs over the face
		 */
		int getFaceUvVertex(int faceIndex, int faceVertexIndex);

		/**
		 * \brief set the UV vertex index over the face
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param faceVertexIndex the index of the vertex within the face, should be in [0..getFaceVertsCount(faceIndex) - 1]
		 * \param uvVertexIndex the UV vertex index, should be in [0..vertsUvCount() - 1]
		 */
		 void setFaceUvVertex(int faceIndex, int faceVertexIndex, int uvVertexIndex);

		/**
		 * \brief get the normal vertex index over the face
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param faceVertexIndex the index of the vertex within the face, should be in [0..getFaceVertsCount(faceIndex) - 1]
		 * \return the normal vertex index, -1 if no normals over the face
		 */
		int getFaceNormalVertex(int faceIndex, int faceVertexIndex);

		/**
		 * \brief set the normal vertex index over the face
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param faceVertexIndex the index of the vertex within the face, should be in [0..getFaceVertsCount(faceIndex) - 1]
		 * \param normalVertexIndex the normal vertex index, should be in [0..vertsNormalCount() - 1]
		 */
		void setFaceNormalVertex(int faceIndex, int faceVertexIndex, int normalVertexIndex);

		/**
		 * \brief get the list of UV vertices indices over the face
		 * \param face the face index
		 * \return the list of UV vertices indices
		 */
		std::vector<int> getFaceUvVerts(int face);
		void getFaceUvVerts(int face, list<int>& vertices);

		/**
		 * \brief get the object index over the face, see the getObjectsCount(), getObjectName()
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \return the object index
		 */
		int getFaceObject(int faceIndex);

		/**
		 * \brief set the object index for the face, see the getObjectsCount(), getObjectName()
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param objectIndex the object index to set for the face
		 */
		void setFaceObject(int faceIndex, int objectIndex);

		/**
		 * \brief get the material index over the face, see the getMaterialsCount(), getMaterialName()
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \return the material index
		 */
		int getFaceMaterial(int faceIndex);

		/**
		 * \brief set the material index over the face, see the getMaterialsCount(), getMaterialName()
		 * \param faceIndex the face index, should be in [0..facesCount() - 1]
		 * \param materialIndex the material index to set for the face
		 */
		void setFaceMaterial(int faceIndex, int materialIndex);

		/**
		 * \brief returns the objects count in the mesh
		 * \return the count
		 */
		int getObjectsCount();

		/**
		 * \brief get the name of the object
		 * \param idx the object index
		 * \return the name
		 */
		std::string getObjectName(int idx);

		/**
		 * \brief set object name
		 * \param idx the object index
		 * \param name the new name
		 */
		void setObjectName(int idx, const std::string& name);

		/**
		 * \brief add new object to the mesh
		 * \param name the object name
		 * \return the object index
		 */
		int addObject(const char* name);

		/**
		 * \brief remove object from the mesh
		 * \param idx the object index
		 */
		void removeObject(int idx);

		/**
		 * \brief unify all objects in the mesh, i.e. make one object
		 * \param name the name of the new object, if empty, the name of the first object will be used
		 */
		void unifyAllObjects(const std::string& name = "");

		/**
		 * \brief get the materials count in the mesh
		 * \return the count
		 */
		int getMaterialsCount();

		/**
		 * \brief add new material to the mesh
		 * \param name the material name
		 * \return the material index
		 */
		int addMaterial(const char* name);

		/**
		 * \brief remove the material (and corresponding faces) from the mesh
		 * \param idx the material index
		 */
		void removeMaterial(int idx);

		/**
		 * \brief get the name of the material
		 * \param idx the material index
		 * \return the name
		 */
		std::string getMaterialName(int idx);

		/**
		 * \brief set material name
		 * \param idx the material index
		 * \param name the new name
		 */
		void setMaterialName(int idx, const std::string& name);

		/**
		 * \brief get the texture name of the material
		 * \param idx the material index
		 * \param texture_layer the texture layer, 0 - color, 1 - gloss, 2 - bump/displacement, 3 - normalmap, 4 - specular color, 5 - emossive (color), 6 - emissive power
		 * \return the texture path (full or relative to 3DCoat documents folder)
		 */
		std::string getMaterialTexture(int idx, int texture_layer);

		/**
		 * \brief set the texture layer filename of the material
		 * \param idx the material index
		 * \param texture_layer the texture layer, 0 - color, 1 - gloss, 2 - bump/displacement, 3 - normalmap, 4 - specular color, 5 - emossive (color), 6 - emissive power
		 * \param texture_path the texture path (full or relative to 3DCoat documents folder)
		 */
		void setMaterialTexture(int idx, int texture_layer, const std::string& texture_path);

		/**
		 * \brief extract the mesh from the volume 
		 * \param v the source volume
		 * \param with_subtree if true, the subtree will be extracted, otherwise the single volume taken
		 * \param all_selected if true, all selected volumes will be extracted, otherwise only the current volume
		 */
		void fromVolume(Volume& v, bool with_subtree = false, bool all_selected = false);

		/**
		 * \brief extract the mesh from the volume and reduce it by the given percent
		 * \param v the source volume
		 * \param reduction_percent 0 means no reduction, 100 means 100% reduction, i.e. the mesh will be reduced to a single triangle
		 * \param with_subtree if true, the subtree will be extracted, otherwise the single volume taken
		 * \param all_selected if true, all selected volumes will be extracted, otherwise only the current volume
		 */
		void fromReducedVolume(Volume& v, float reduction_percent, bool with_subtree = false, bool all_selected = false);

		/**
		 * \brief extract the mesh from the volume and reduce to the given polycount
		 * \param v the source volume
		 * \param max_polycount the required polycount
		 * \param with_subtree if true, the subtree will be extracted, otherwise the single volume taken
		 * \param all_selected if true, all selected volumes will be extracted, otherwise only the current volume
		 */
		void fromVolumeWithMaxPolycount(Volume& v, int max_polycount, bool with_subtree = false, bool all_selected = false);

		/**
		 * \brief merge this mesh to the volume object
		 * \param v the destination volume
		 * \param transform the applied transformation
		 * \param op the boolean operation to be performed, -1 means no operation, raw merge, 0 - 1, 1 - subtract, 2 - intersect
		 */
		void toVolume(Volume& v, const mat4& transform = mat4::Identity, BoolOpType op = BOOL_MERGE);

		/**
		 * \brief insert without boolean operation, if the volume is not in surface mode (volumetric) the boolean ADD will be performed anyway
		 * \param v the destination volume
		 * \param transform the transform
		 */
		void insertInVolume(Volume& v, const mat4& transform = mat4::Identity);

		/**
		 * \brief boolean add to volume
		 * \param v the destination volume
		 * \param transform the transform
		 */
		void addToVolume(Volume& v, const mat4& transform = mat4::Identity);

		/**
		 * \brief boolean subtraction of the mesh from the volume
		 * \param v the destination volume
		 * \param transform the transform
		 */
		void subtractFromVolume(Volume& v, const mat4& transform = mat4::Identity);

		/**
		 * \brief boolean intersection of the mesh with the volume
		 * \param v the destination volume
		 * \param transform the transform
		 */
		void intersectWithVolume(Volume& v, const mat4& transform = mat4::Identity);

		/**
		 * \brief take the whole mesh from the retopo room
		 */
		void fromRetopo();

		/**
		 * \brief get the mesh from the paint room
		 */
		void fromPaintRoom();

		/**
		 * \brief reduce the mesh to the given polycount, mesh will be triangulated
		 * \param destination_triangles_count the required triangles count, if it is above the existing, nothing happens
		 */
		void reduceToPolycount(int destination_triangles_count);

		/**
		 * \brief triangulate the mesh
		 */
		void triangulate();

		/**
		 * \brief Perform the boolean operation with the given mesh
		 * \param With the mesh to perform the operation with over the current mesh
		 * \param op the operation, see BoolOpType (-1 means no operation, 0 - add, 1 - subtract, 2 - intersect)
		 */
		void booleanOp(Mesh& With, BoolOpType op);

		/**
		 * \brief get the list of all positional vertices of the mesh
		 * \return the list of vec3
		 */
		std::vector<vec3> getMeshVertices();

		/**
		 * \brief get the list of all normal vertices of the mesh
		 * \return the list of vec3
		 */
		std::vector<vec3> getMeshNormals();

		/**
		 * \brief get the list of all UV vertices of the mesh
		 * \return the list of vec2
		 */
		std::vector<vec2> getMeshUVs();

		/**
		 * \brief set the list of all positional vertices for the mesh
		 * \param positions the list of positions
		 */
		void setMeshVertices(const std::vector<vec3>& positions);

		/**
		 * \brief set the list of all normal vertices for the mesh
		 * \param normals the list of normals (vec3)
		 */
		void setMeshNormals(const std::vector<vec3>& normals);

		/**
		 * \brief set the list of all UV vertices for the mesh
		 * \param uvs the list of UVs (vec2)
		 */
		void setMeshUVs(const std::vector<vec2>& uvs);

		/**
		 * \brief set the complete list of faces for the mesh
		 * \param faces the format of faces is:\n
		 * amount_ot_vets_in_face1, vertex1_face1, vertex2_face1...vertexN-1_face1,\n
		 * amount_ot_vets_in_face2, vertex1_face2, vertex2_face2...\n
		 * ...
		 */
		void setMeshFaces(const std::vector<int>& faces);

		/**
		 * \brief add the list of all positional vertices for the mesh
		 * \param positions the list of positions
		 */
		void addMeshVertices(const std::vector<vec3>& positions);

		/**
		 * \brief add the list of all normal vertices for the mesh
		 * \param normals the list of normals (vec3)
		 */
		void addMeshNormals(const std::vector<vec3>& normals);

		/**
		 * \brief add the list of all UV vertices for the mesh
		 * \param uvs the list of UVs (vec2)
		 */
		void addMeshUVs(const std::vector<vec2>& uvs);

		/**
		 * \brief add the list of faces for the mesh, pay attention, all vertex indices are global over the whole mesh!
		 * \param faces the format of faces is:\n
		 * amount_ot_vets_in_face1, vertex1_face1, vertex2_face1...vertexN-1_face1,\n
		 * amount_ot_vets_in_face2, vertex1_face2, vertex2_face2...\n
		 * ...
		 */
		void addMeshFaces(const std::vector<int>& faces);

		/**
		 * \brief clear all positional vertices of the mesh
		 */
		void clearVerts();

		/**
		 * \brief clear all uv vertices of the mesh
		 */
		void clearUvVerts();

		/**
		 * \brief clear all normal vertices of the mesh
		 */
		void clearNormals();

		/**
		 * \brief clear all faces of the mesh
		 */
		void clearFaces();

		/**
		 * \brief remove the set of vertices from the mesh
		 * \param faces the list of faces indices to remove
		 */
		void removeFaces(const std::vector<int>& faces);

		/**
		 * \brief clear all objects
		 */
		void clearObject();

		/**
		 * \brief clear all materials
		 */
		void clearMaterials();

		/**
		 * \brief ensure that at least one material and one object exist in the mesh
		 */
		void ensureMaterialsAndObjectsExist();

		/**
		 * \brief add the named object
		 * \param name the name for the object
		 * \return the index of new object in the objects list
		 */
		int addObject(const std::string& name);

		/**
		 * \brief add the named material
		 * \param name the name for the material
		 * \return the index of new material in the materials list
		 */
		int addMaterial(const std::string& name);

		/**
		 * \brief remove all unused objects and materials
		 */
		void removeUnusedObjectsAndMaterials();

		/**
		 * \brief remove all unused vertices
		 */
		void removeUnusedVerts();

		/**
		 * \brief remove all faces that contain zero vertices
		 */
		void removeUnusedFaces();

		/**
		 * \brief Cut off the mesh by the plane, the result is stored in the current mesh, the part of the mesh that is on the side of the negative normal direction is removed
		 * \param start the start point of the plane
		 * \param NormalDirection the normal direction of the plane
		 */
		void cutByPlane(const vec3& start, const vec3& NormalDirection);

		/**
		 * \brief Cut off the mesh by the distorted plane (using the Perlin noise), the result is stored in the current mesh, the part of the mesh that is on the side of the negative normal direction is removed
		 * \param start the start point of the plane
		 * \param NormalDirection the normal direction of the plane
		 * \param noise_degree the degree of the noise
		 * \param noise_scale the scale of the noise
		 * \param seed the seed for the noise
		 */
		void cutByDistortedPlane(const vec3& start, const vec3& NormalDirection, float noise_degree, float noise_scale, int seed = 0);

		/**
		 * \brief distort the mesh by the Perlin noise
		 * \param noise_degree the degree of the noise
		 * \param noise_scale the scale of the noise
		 * \param anisotropic if false, the noise will be applied in the direction of the normals, othervice the noise directed in random direction regardless the normals
		 * \param seed the seed for the noise
		 */
		void distortByPerlinNoise(float noise_degree, float noise_scale, bool anisotropic = false, int seed = 0);

		/**
		 * \brief split the mesh into disconnected parts
		 * \return the list of meshes
		 */
		std::vector<Mesh> splitDisconnectedParts();

		/**
		 * \brief apply symmetry to the mesh
		 * \param start the start point of the plane
		 * \param NormalDirection the negative part (regarding the plane normal) of the mesh is removed, replaced with positive part
		 * \param resultInQuads the cut faces will produce quads instead of triangles
		 */
		void symmetry(const vec3& start, const vec3& NormalDirection, bool resultInQuads);

		/**
		 * \brief Detect the symmetry planes of the mesh
		 * \return the list of planes
		 */
		std::vector<comms::cPlane> autodetectSymmetryPlanes();

		/**
		 * \brief weld the mesh, remove all vertices that are closer than minimal_relative_distance*mesh_bound_box_diagonal to each other
		 * \param minimal_relative_distance the minimal distance between vertices, relative to the mesh bound box diagonal
		 */
		void weld(float minimal_relative_distance = 0.0001f);

		/**
		 * \brief get the mesh bound box
		 * \return the bound box
		 */
		boundbox getBounds() const;

		/**
		 * \brief get the volume of the mesh
		 * \return the volume
		 */
		float getVolume() const;

		/**
		 * \brief calculate the volume even if the mesh is not closed, in this case we define plane that limits the integration
		 * \param start the point on that plane
		 * \param dir the normalized vector, normal to the plane
		 * \return the volume
		 */
		float getOpenSurfaceVolume(const vec3& start, const vec3& dir) const;

		/**
		 * \brief get square of the mesh
		 * \return the square (area)
		 */
		float getSquare() const;

		/**
		 * \brief get the squareof the face
		 * \param face the face index
		 * \return the square
		 */
		float getFaceSquare(int face);

		/**
		 * \brief get the face square in UV space
		 * \param face the face index
		 * \return the square
		 */
		float getFaceUVSquare(int face);

		/**
		 * \brief get the face normal
		 * \param face the face index
		 * \return the face normal
		 */
		vec3 getFaceNormal(int face);

		/**
		 * \brief relax the mesh, keep the vertices count
		 * \param degree the degree of relax, may be  > 1
		 * \param tangent should be tangent relax
		 * \param crease_angle the crease angle between faces (degrees), if the angle between faces is less than crease_angle, the edge relaxed
		 */
		void relax(float degree, bool tangent, float crease_angle = 180);

		/**
		 * \brief create the box mesh
		 * \param center the box center
		 * \param size the box size
		 * \param xAxis the x-axis direction, if zero, the x-axis is default - (1,0,0)
		 * \param yAxis the y-axis direction, if zero, the y-axis is default - (0,1,0)
		 * \param zAxis the z-axis direction, if zero, the z-axis is default - (0,0,1)
		 * \param detail_size the average length of the edge over the figure. The figure will be divided so that edges length will be approximately the detail_size
		 * \param fillet the fillet radius
		 * \param nx the number of segments along the x-axis (if all of nx, ny, nz are above zero, it overrides the detail_size)
		 * \param ny the number of segments along the y-axis (if all of nx, ny, nz are above zero, it overrides the detail_size)
		 * \param nz the number of segments along the z-axis (if all of nx, ny, nz are above zero, it overrides the detail_size)
		 * \return the box mesh
		 */
		static Mesh box(const vec3& center = vec3::Zero, const vec3& size = vec3::One, const vec3& xAxis =vec3::Zero, const vec3& yAxis = vec3::Zero, const vec3& zAxis = vec3::Zero, float detail_size = 1, float fillet = 0.0f, int nx=0, int ny=0, int nz=0);

		/**
		 * \brief create the sphere mesh
		 * \param center the sphere center
		 * \param radius the sphere radius
		 * \param detail_size the average length of the edge over the figure. The figure will be divided so that edges length will be approximately the detail_size
		 * \return the sphere mesh
		 */
		static Mesh sphere(const vec3& center = vec3::Zero, float radius = 1.0f, float detail_size = 1);

		/**
		 * \brief create the cylinder mesh
		 * \param center the center of the cylinder
		 * \param radius the radius of the cylinder
		 * \param height the height of the cylinder
		 * \param detail_size the average length of the edge over the figure. The figure will be divided so that edges length will be approximately the detail_size
		 * \param slices the number of slices, it overrides the detail_size if all of slices, caps, rings are above zero
		 * \param caps the number of caps, it overrides the detail_size if all of slices, caps, rings are above zero
		 * \param rings the number of rings, it overrides the detail_size if all of slices, caps, rings are above zero
		 * \param fillet the fillet radius
		 * \return the cylinder mesh
		 */
		static Mesh cylinder(const vec3& center = vec3::Zero, float radius = 1, float height = 2, float detail_size = 1, int slices = 0, int caps = 0, int rings = 0, float fillet = 0);

		/**
		 * \brief create the cone mesh
		 * \param center the center of the cone (the cone base center)
		 * \param radius the cone radius
		 * \param height the cone height
		 * \param detail_size the average length of the edge over the figure. The figure will be divided so that edges length will be approximately the detail_size
		 * \param topAxis the top axis direction, if zero, the top axis is default - (0,1,0)
		 * \return the cone mesh
		 */
		static Mesh cone(const vec3& center = vec3::Zero, float radius = 1, float height = 2, float detail_size = 1, const vec3& topAxis = vec3::AxisY);

		/**
		 * \brief create the single-side plane mesh, the faces normals are put toward the vec3.Cross(xAxis, yAxis)
		 * \param center the center of the plane
		 * \param sizeX the plane size along the X-axis
		 * \param sizeY the plane size along the Y-axis
		 * \param divisionsX amount of divisions along the X-axis 
		 * \param divisionsY amount of divisions along the Y-axis
		 * \param xAxis the vector of the X-axis
		 * \param yAxis the vector of the Y-axis
		 * \return the plane mesh
		 */
		static Mesh plane(const vec3& center = vec3::Zero, float sizeX = 2, float sizeY = 2, int divisionsX = 2, int divisionsY = 2, const vec3& xAxis = vec3::AxisX, const vec3& yAxis = vec3::AxisY);

		/**
		 * \brief create the single-side triangular plane mesh that consists mostly of quasi equally-sided triangles 
		 * \param center the center of the plane
		 * \param sizeX the plane size along the X-axis
		 * \param sizeY the plane size along the Y-axis
		 * \param divisionsX amount of divisions along the X-axis
		 * \param divisionsY amount of divisions along the Y-axis
		 * \param xAxis the vector of the X-axis
		 * \param yAxis the vector of the Y-axis
		 * \return the hexagonal plane mesh
		 */
		static Mesh hexagonal_plane(const vec3& center = vec3::Zero, float sizeX = 2, float sizeY = 2, int divisionsX = 2, int divisionsY = 2, const vec3& xAxis = vec3::AxisX, const vec3& yAxis = vec3::AxisY);

		/**
		 * \brief Create the text mesh
		 * \param string the text string
		 * \param font the font name
		 * \param height the text height
		 * \param center the text center
		 * \param text_direction the text direction left to right
		 * \param text_normal the normal direction of the text
		 * \param thickness the thickness of the text
		 * \param align the text align, 0 - left, 1 - center, 2 - right
		 * \return the text mesh
		 */
		static Mesh text(const char* string, const char* font = "tahoma", float height = 10.0f, const vec3 & center = vec3::Zero, const vec3& text_direction = vec3::AxisX, const vec3& text_normal = vec3::AxisY, float thickness = 1, int align = 1);

		/**
		 * \brief Create the vector displacement map from the mesh and save it as EXR file. The mesh is put on plane at center and clamped by that plane.
		 * \param side the EXR  file side size
		 * \param path_to_exr the path to the EXR file
		 * \param center the center of the plane
		 * \param radius the radius that should include the mesh
		 * \param up the up vector of the plane
		 * \param x the x vector of the plane
		 * \param y the y vector of the plane
		 */
		void createVDM(int side, const char* path_to_exr, const vec3& center = vec3::Zero, float radius = 1, const vec3& up = vec3::AxisZ, const vec3& x = vec3::AxisX, const vec3& y = vec3::AxisY);

		/**
		 * \brief add some thickness to the mesh (intrude a bit)
		 * \param thickness_out the thickness in the outer direction (extrusion)
		 * \param thickness_in the thickness in the inner direction (intrusion)
		 * \param divisions the amount of divisions of the edge
		 */
		void shell(float thickness_out, float thickness_in, int divisions = 1);

		/**
		 * \brief extrude open edges of the mesh
		 * \param distance the distance to extrude
		 * \param direction the extrude direction, if zero , the direction is the local vertex normal
		 * \returns the list of extruded edges, even is the start vertex, odd is the end vertex
		 */
		std::vector<int> extrudeOpenEdges(float distance, vec3 direction = vec3::Zero);

		/**
		 * \brief extrude open edges of the mesh
		 * \param distance the distance to extrude
		 * \returns the list of extruded edges, even is the start vertex, odd is the end vertex
		 */
		std::vector<int> expandOpenEdges(float distance);

		/**
		 * \brief get the list of open edges
		 * \return the list of open edges, the even is the start vertex, the odd is the end vertex
		 */
		std::vector<int> getOpenEdges();

		/**
		 * \brief get the mesh size along some axis
		 * \param dir the axis direction
		 * \return the size along the axis
		 */
		float getLengthAlongDirection(const vec3& dir) const;

		/**
		 * \brief calculate the center mass of the mesh
		 * \return the center mass of the surface
		 */
		vec3 getCenterMass() const;

	
	protected:
		void setMesh(const comms::cMeshContainer& mesh) {
			if (MeshData) {
				MeshData->Clear();
				MeshData->Copy(mesh);
			}
			else {
				MeshData = new comms::cMeshContainer(mesh);
				inc_ref(MeshData);
			}
		}
		friend class prim;
	};

	/**
	 * \brief The image references. Look the cImage for the list of allowed operations
	 */
	class APICALL Image : public comms::cImage {
	public:
		Image();
		Image(const Image& im);
		/**
		 * \brief Read the image from the file
		 * \param name the image name
		 * \return true if loaded successfully
		 */
		bool Read(const char* name);

		/**
		 * \brief Write the image to file
		 * \param name the filename
		 * \return true if succeed
		 */
		bool Write(const char* name);

		/**
		 * \brief Get image from texture
		 * \param texture id
		 * \return true if succeed
		 */
		bool FromTexture(int texture_id);

		/**
		 * \brief Create texture from image
		 * \return true if succeed
		 */
		int ToTexture();
#ifdef PYBIND11_DEF
		/**
		 * \brief paste image to image
		 * \param numpy array for source
		 * \param pasteLeft
		 * \param pasteTop
		 * \param cropLeft
		 * \param cropTop
		 * \param cropRight
		 * \param cropBottom
		 * \param flipY
		 * \return true if succeed
		 */
		int Paste(PY_BYTE_ARRAY src_data, int pasteLeft = 0, int pasteTop = 0, int cropLeft = 0, int cropTop = 0, int cropRight = 0, int cropBottom = 0, bool flipY = false);

		/**
		 * \brief Pointer to the data
		 * \return true if succeed
		 */
		size_t Pointer();

		/**
		 * \brief Get image from texture
		 * \param numpy array for source
		 * \return true if succeed
		 */
		bool FromArray(PY_BYTE_ARRAY src_data);
		bool FromArray(PY_DWORD_ARRAY src_data);
		bool FromArray(PY_FLOAT_ARRAY src_data);

		pybind11::buffer_info _py_buffer_info();
#endif
	};

	class Volume;

	class APICALL symm {
	public:
		/**
		 * \brief Enable the symmetry
		 * \param _enable true to enable, false to disable
		 * \return reference for the chain-like operations 
		 */
		static symm& enable(bool _enable = true);

		static bool enabled();

		/**
		 * \brief disable the symmetry
		 * \return reference
		 */
		static symm& disable();

		/**
		 * \brief Enable the XYZ-mirror symmetry
		 * \param x true to enable x-symmetry, false to disable
		 * \param y true to enable y-symmetry, false to disable
		 * \param z true to enable z-symmetry, false to disable
		 * \return reference
		 */
		static symm& xyz(bool x, bool y, bool z);

		/**
		 * \brief check if the XYZ symmetry enabled
		 * \return true if this type of the symmetry active
		 */
		static bool is_xyz();

		/**
		 * \brief check x symmetry state
		 * \return reference to the x symmetry state  
		 */
		static bool& x();

		/**
		 * \brief check y symmetry state
		 * \return reference to the y symmetry state
		 */
		static bool& y();

		/**
		 * \brief check z symmetry state
		 * \return reference to the z symmetry state
		 */
		static bool& z();

		/**
		 * \brief Enable the axial symmetry
		 * \param n the order of the axial symmetry
		 * \param extraMirror add the extra mirror orthogonal to the axis
		 * \param stepSymmetry enable the step symmetry
		 * \return reference
		 */
		static symm& axial(int n, bool extraMirror = false, bool stepSymmetry = false);

		/**
		 * \brief Check if the axial symmetry enabled		
		 * \return true if the axial symmetry enabled
		 */
		static bool is_axial();

		/**
		 * \brief returns the axial symmetry order if axial or axial mirror symmetry enabled
		 * \return the reference to the order of the axial symmetry
		 */
		static int& axialOrder();

		/**
		 * \brief returns the state of extra mirror, this is valid only tor the axial symmetry
		 * \return the reference to the extra mirror state
		 */
		static bool& extraMirror();

		/**
		 * \brief returns the state of step symmetry
		 * \return the reference to the step symmetry state
		 */
		static bool& stepSymmetry();
		/**
		 * \brief Enable the axial mirror symmetry
		 * \param n the order of the symmetry
		 * \param extraMirror dd the extra mirror orthogonal to the axis 
		 * \param stepSymmetry enable the step symmetry
		 * \return the reference
		 */
		static symm& axialMirror(int n, bool extraMirror = false, bool stepSymmetry = false);

		/**
		 * \brief Check if the axial mirror symmetry enabled
		 * \return true if the axial mirror symmetry enabled
		 */
		static bool isAxialMirror();

		/**
		 * \brief Enable the translation symmetry
		 * \param numX number of x-repeats
		 * \param stepX the step of the x-repeat
		 * \param numY number of y-repeats
		 * \param stepY the step of the y-repeat
		 * \param numZ number of z-repeats
		 * \param stepZ the step of the z-repeat
		 * \return the reference
		 */
		static symm& translation(int numX, float stepX, int numY, float stepY, int numZ, float stepZ);

		/**
		 * \brief Check if the translation symmetry enabled
		 * \return the state
		 */
		static bool is_translation();

		/**
		 * \brief returns the reference to the number of the x repeats if the translational symmetry used
		 * \return the bool reference
		 */
		static int& numX();

		/**
		 * \brief returns the reference to the x-step if the translational symmetry used
		 * \return the value reference
		 */
		static float& stepX();

		/**
		 * \brief returns the reference to the number of the y repeats if the translational symmetry used
		 * \return the bool reference
		 */
		static int& numY();

		/**
		 * \brief returns the reference to the y-step if the translational symmetry used
		 * \return the value reference
		 */
		static float& stepY();

		/**
		 * \brief returns the reference to the number of the z repeats if the translational symmetry used
		 * \return the bool reference
		 */
		static int& numZ();

		/**
		 * \brief returns the reference to the z-step if the translational symmetry used
		 * \return the value reference
		 */
		static float& stepZ();

		/**
		 * \brief set the symmetry to be in global space
		 * \return the reference
		 */
		static symm& toGlobalSpace();

		/**
		 * \brief set the symmetry to be in local space
		 * \return the reference
		 */
		static symm& toLocalSpace();

		/**
		 * \brief set the symmetry to general case
		 * \return the reference
		 */
		static symm& toGeneral();

		/**
		 * \brief set the central point for the symmetry
		 * \param pos the position (in local or global space, see the localSpace() or globalSpace())
		 * \return the reference
		 */
		static symm& set_start(const vec3& pos);

		/**
		 * \brief get the start point reference
		 * \return the point reference
		 */
		static vec3& start();

		/**
		 * \brief set the end point for the symmetry axis, calling this function enables the general case of the symmetry
		 * \param pos the position
		 * \return the reference
		 */
		static symm& set_end(const vec3& pos);

		/**
		 * \brief the end point reference
		 * \return the point reference
		 */
		static vec3& end();

		/**
		 * \brief Show or hide the symmetry planes
		 * \param show set true to show
		 * \return the reference
		 */
		static symm& showSymmetryPlane(bool show = true);

		/**
		 * \brief enable the custom symmetry, provide the symmetry transfoms
		 * \param symmetryTransforms the list of additional transforms that will be applied to the any user action
		 * \return the reference
		 */
		static symm& setCustomSymetryTransforms(list<mat4>& symmetryTransforms);
#ifdef PYBIND11_DEF
		/**
		 * \brief enable the custom symmetry, provide the symmetry transfoms
		 * \param symmetryTransforms the list of additional transforms (list of coat.mat4) that will be applied to the any user action
		 * \return the reference
		 */
		static symm& setCustomSymetryTransforms(pybind11::list& symmetryTransforms);
#endif

		/**
		 * \brief Check if the custom symmetry used
		 * \return true if the custom symmetry enabled
		 */
		static bool isCustomSymmetry();

		/**
		 * \brief Returns all transforms using the current symmetry state
		 * \param symmetryTransforms the resulting list
		 */
		static symm& getCurrentTransforms(list<mat4>& symmetryTransforms);

		/**
		 * \brief Returns all transforms using the current symmetry state
		 * \return the resulting list of coat.mat4
		 */
		static std::vector<mat4> getCurrentTransforms();

		/**
		 * \brief Returns all symmetry planes using the current symmetry state
		 * \param planes the resulting list 
		 */
		static symm& getCurrentPlanes(list<comms::cPlane>& planes);

		/**
		 * \brief Returns all symmetry planes using the current symmetry state
		 * \return the resulting list of planes (coat.plane)
		 */
		static std::vector<comms::cPlane> getCurrentPlanes();

		/**
		* \brief Totally disable symmetry, don't forget to enable after all operations!
		*/
		static void disableGlobally();

		/**
		* \brief Enable symmetry (preliminary disabled by disableGlobally)
		*/
		static void enableGlobally();


	};
	/**
	 * \brief The scene element, like sculpt object or curve
	 */
	class APICALL SceneElement {
	protected:
		ItemsTree* el;
		VoxTreeBranch* tb() const;
		OneCurveObject* cu() const;
		friend class Scene;
		friend class Curve;
		friend class Volume;
	public:
		SceneElement();		
		SceneElement(VolumeObject* vo);
		SceneElement(ItemsTree* c);
		SceneElement(const SceneElement& other);

		~SceneElement();

		const bool operator==(const SceneElement& other) const;
		const bool operator!=(const SceneElement& other) const;

		/**
		 * \brief get the parent scene graph element
		 * \return the parent reference
		 */
		SceneElement parent() const;

		/**
		 * \brief returns the child elements count
		 * \return child count
		 */
		int childCount() const;

		/**
		 * \brief returns child element by index
		 * \param index the index of the element in subtree
		 * \return the child reference
		 */
		SceneElement child(int index) const ;

		/**
		 * \brief Check if it is the sculpt object
		 * \return true if this is the sculpt object
		 */
		bool isSculptObject() const;

		/**
		 * \brief Check if the element is curve
		 * \return true if this is curve
		 */
		bool isCurve() const;

		/**
		 * \brief Set the transform matrix
		 * \param Transform the transform matrix
		 * \return this element reference
		 */
		const SceneElement& setTransform(const mat4& Transform) const ;

		/**
		 * \brief Additional transform over the object
		 * \param Transform the matrix
		 * \return this element reference
		 */
		const SceneElement& transform(const mat4& Transform) const ;

		/**
		 * \brief this command useful if you use voxels, it sets the scale for the volume so that there will be density_value of voxels per mm
		 * \param density_value the voxels per mm
		 */
		const SceneElement& density(float density_value) const;

		/**
		 * \brief Additional transform over the object, not applied to child objects
		 * \param Transform the matrix
		 * \return this element reference
		 */
		const SceneElement& transform_single(const mat4& Transform) const;

		/**
		 * \brief get the scene element transform
		 * \return the transform matrix
		 */
		mat4 getTransform() const ;

		/**
		 * \brief Clear the element content
		 * \return this element reference
		 */
		const SceneElement& clear() const ;

		/**
		 * \brief get the element name
		 * \return the name
		 */
		const char* name() const;

		/**
		 * \brief get the linked file path
		 * \return the name
		 */
		const char* getLinkedPath(int id) const;

		/**
		 * \brief get the linked file path
		 * \return the name
		 */
		int linkedObjectCount() const;

		/**
		 * \brief set the linked file path
		 * \return the name
		 */
		void addLinkedPath(const char* path) const;

		/**
		 * \brief rename the element
		 * \param name the new name
		 * \return this element reference
		 */
		const SceneElement& rename(const char* name) const;

		/**
		 * \brief add the child element of the same nature
		 * \param name the name
		 * \return the new element reference
		 */
		SceneElement addChild(const char* name) const;

		/**
		 * \brief find the element in subtree by name
		 * \param name the name t seek
		 * \return the element reference
		 */
		SceneElement findInSubtree(const char* name) const;

		/**
		 * \brief iterate over the subtree
		 * \param fn the function to call, return true if need to stop the iterations
		 * \return true if the callback interrupted the iterations
		 */
		bool iterateSubtree(const std::function<bool(SceneElement)>& fn) const;
#ifdef PYBIND11_DEF
		/**
		 * \brief iterate over the subtree
		 * \param fn the function to call, return true if need to stop the iterations,
		 * function looks like\n
		 * \code
		 * def fn(el):
		 *     ...code...
		 *     return False or True
		 * \endcode
		 * el is coat.SceneElement
		 * \return true if the callback interrupted the iterations
		 */
		bool iterateSubtree(const pybind11::object& fn) const;
#endif

		/**
		 * \brief iterate over the visible subtree
		 * \param fn the function to call, return true if need to stop the iterations
		 * \return true if the callback interrupted the iterations
		 */
		bool iterateVisibleSubtree(const std::function<bool(SceneElement)>& fn) const;
#ifdef PYBIND11_DEF
		/**
		 * \brief iterate over the visible subtree
		 * \param fn the function to call, return true if need to stop the iterations,
		 * function looks like\n
		 * \code
		 * def fn(el):
		 *     ...code...
		 *     return False or True
		 * \endcode
		 * el is coat.SceneElement
		 * \return True if the callback interrupted the iterations
		 */
		bool iterateVisibleSubtree(const pybind11::object& fn) const;
#endif

		/**
		 * \brief merge all subtree volumes into this
		 * \param booleanMerge use boolean summ to merge. Othervice merge meshes without booleans.
		 * This option works only for surfave, in voxels it will always do boolean summ
		 */
		void mergeSubtree(bool booleanMerge = false) const;

		/**
		 * \brief merge the volume to another one, delete this volume
		 * \param dest the destination
		 * \param op the boolean operation type
		 */
		void mergeTo(const SceneElement& dest, BoolOpType op);

		/**
		 * \brief copy and merge the volume to another one, delete this volume
		 * \param dest the destination
		 * \param op the boolean operation type
		 */
		void copyMergeTo(SceneElement& dest, BoolOpType op);

		/**
		 * \brief remove the whole subtree
		 */
		void removeSubtree() const;

		/**
		 * \brief remove one child from the subtree
		 * \param index index of the child
		 */
		void removeSubtreeItem(int index) const;

		/**
		 * \brief remove this item and all child objects from the scene
		 */
		void remove() const;

		/**
		 * \brief diplicate the item
		 * \return the new item reference
		 */
		SceneElement duplicate() const;

		/**
		 * \brief create the instance of the object if instancing supported
		 * \return the instance reference
		 */
		SceneElement duplicateAsInstance() const;

		/**
		 * \brief change the parent element for the current one
		 * \param newParent the new parent reference. Pay attention, changing paren is not always possible!
		 */
		void changeParent(SceneElement newParent) const;

		/**
		 * \brief move the element to another parent
		 * \param newParent the new parent reference
		 * \param indexInParent the index in the parent
		*/
		void moveTo(SceneElement newParent, int indexInParent) const;

		/**
		 * \brief check if the element is parent of another one
		 * \param child the child element
		 * \return true if this element is parent of the child
		 */
		bool isParentOf(SceneElement child) const;

		/**
		 * \brief returns own visibility state reference. It does not take into account that parent may be invisible.
		 * \return item local visibility reference, you may get and set the visibility with the reference.
		 */
		bool& visible() const;

		/**
		 * \brief set the visibility of the element
		 * \param visible true if need to be visible
		 */
		void setVisibility(bool visible) const;

		/**
		 * \brief returs the state of ghosting (if available)
		 * \return true if ghosted
		 */
		bool ghost() const;

		/**
		 * \brief sets the ghosting state (if available)
		 * \param ghost set true to ghost
		 */
		void setGhost(bool ghost) const;

		/**
		 * \brief get the reference color for the element
		 * \return the color (r,g,b,a), each channel is 0..1
		 */
		vec4 getReferenceColor();

		/**
		 * \brief set the reference color for the element
		 * \param color the (r, g, b, a) color, each channel is 0..1
		 */
		void setReferenceColor(const vec4& color);

		/// returns the volume object to operate over voxels or surface
		Volume Volume() const;

		/// add the object to selected
		void select() const;

		/// unselect all similar elements and select this one
		void selectOne() const;

		/// unselect all similar objects
		void unselectAll() const;

		/// Check if the scene element is selected
		bool selected();

		/// Collect the selected elements in the subtree (including this element if selected)
		void collectSelected(list<SceneElement>& elemList);

		/// Collect the selected elements in the subtree (including this element if selected)
		std::vector<SceneElement> collectSelected();
	};

	class VolumeCache {
	public:
		VolumeCache();
		tri_DWORD CellID;
		VolumeCell* pCell;
	};

	/**
	 * \brief The class allows to operate over voxels/surface on the relatively low-level
	 */
	class APICALL Volume {
	protected:
		VoxTreeBranch* tb;
		VolumeObject* Obj;
		friend class SceneElement;
	public:
		Volume();
		Volume(VoxTreeBranch* tb);
		Volume(VolumeObject* vo);
		Volume(const Volume& vol);
		/**
		 * \brief checks if object is valid
		 * \return true if the volume exists
		 */
		bool valid() const;

		/**
		 * \brief Check if in surface mode
		 * \return true if in surface mode
		 */
		bool isSurface() const;

		/**
		 * \brief Check if in voxel mode
		 * \return true if in voxel mode
		 */
		bool isVoxelized() const;

		/**
		 * \brief turn to surface mode, the triangles will be tangentially relaxed
		 */
		void toSurface();

		/**
		 * \brief turn to voxels, auto-voxelize
		 */
		void toVoxels();

		/**
		 * \brief enable or disable the voxel-based coloring. It is applied wherever possible - merging models, brushing, creating parametric voxel figures, etc
		 * \param enable true to enable
		 */
		static void enableVoxelsColoring(bool enable = true);

		/**
		 * \brief set the default color to fill voxels if the voxel coloring enabled
		 */
		static void color(DWORD CL);

		/**
		 * \brief assign the color for the voxel operations
		 * \param r red value 0..255
		 * \param g green value 0..255
		 * \param b blue value 0..255
		 * \param a alpha value 0..255
		 */
		static void color(float r, float g, float b, float a);

		/**
		 * \brief assign the color for the voxel operations
		 * \param r red value 0..255
		 * \param g green value 0..255
		 * \param b blue value 0..255
		*/
		static void color(float r, float g, float b);

		/**
		 * \brief assign the color for the voxel operations
		 * \param colorid the color in any suitable form: "RGB", "ARGB", "RRGGBB", "AARRGGBB", "#RGB", "#ARGB", "#RRGGBB", "#AARRGGBB",
		 * any web-color common name as "red", "green", "purple", google "webcolors"
		 */
		static void color(const char* colorid);

		/**
		 * \brief assign the gloss for the voxel operations, it will work only if the color already assigned
		 * \param value the [0..1] value of the gloss
		 */
		static void gloss(float value);

		/**
		 * \brief assign the roughness for the voxel operations, it will work only if the color already assigned
		 * \param value the [0..1] value of the roughness
		 */
		static void roughness(float value);

		/**
		 * \brief the metalliclty value for the voxel operations, it will work only if the color already assigned
		 * \param value the [0..1] metal value
		 */
		static void metal(float value);

		/**
		 * \brief merge the mesh into scene
		 * \param mesh the Mesh reference
		 * \param transform the transform applied
		 * \param op the type of the merge
		 */
		void mergeMesh(Mesh& mesh, const mat4& transform = mat4::Identity, BoolOpType op = BOOL_MERGE);

		/**
		 * \brief insert the mesh into the volume, in case of voxels this is identical to addMesh, in case of surface, mesh will be inserted without booleans
		 * \param mesh the mesh reference
		 * \param transform the transform applied
		 */
		void insertMesh(Mesh& mesh, const mat4& transform = mat4::Identity);

		/**
		 * \brief add the mesh to volume (boolean)
		 * \param mesh the mesh reference
		 * \param transform the transform applied
		 */
		void addMesh(Mesh& mesh, const mat4& transform = mat4::Identity);

		/**
		 * \brief subtract the mesh from volume (boolean)
		 * \param mesh the mesh reference
		 * \param transform the transform applied
		 */
		void subtractMesh(Mesh& mesh, const mat4& transform = mat4::Identity);

		/**
		 * \brief intersect the volume with the mesh (boolean)
		 * \param mesh the mesh reference
		 * \param transform the transform applied
		 */
		void intersectWithMesh(Mesh& mesh, const mat4& transform = mat4::Identity);

		/**
		 * \brief merge the mesh with facture, the volume polygons will be hidden, just the texture will be shown (like leafs in TreesGenerator)
		 * \param mesh the mesh that refers texture
		 * \param transform the transform applied
		 * \param op the boolean operation
		 */
		void mergeMeshWithTexture(Mesh& mesh, const mat4& transform = mat4::Identity, BoolOpType op = BOOL_MERGE);

		/**
		* \brief This function is fast (generally realtime) way to fill the volume with voxels. Voxel Brush Engune completely based on this routine.
		* \param densityFunction the function should return the value 0..1, where 0 means empty, 1-filled, 0.5 means we are on the surface.
		* Be careful, function should be limited in space. Coordinates passed in volume's local space.
		* \param growCenters the list of points where function is not zero - that are grow centers
		* \param Subtract true if you need to subtract, in this case you return value that will be subtracted from the volume
		* \param useTempLocation if true all your modifications will be applied after you will finish all chnages
		* \param overHidden if true the function will work as Hide tool (Subtract taken into account)
		* \param useColor if true the voxel color will be used 
		* \return returns itself for the chain-like reference.
		*/
		Volume& makeVoxelFigure(std::function<float(vec3)> densityFunction, const list<vec3>& growCenters, bool Subtract = false, bool useTempLocation = false, bool overHidden = false, bool useColor = false);

		/**
		 * \brief returns the exact voxel density in local space at the exact integer location
		 * \param x X-coordinate
		 * \param y Y-coordinate (up)
		 * \param z Z-coordinate
		 * \param fromBackup take the values from the backup (kept before the modifications started)
		 * \param cache_ref define the variable coat::VolumeCache and pass there (in same thread) to speed up access;
		 * \return the density 0..1
		 */
		float getExactDencity(int x, int y, int z, bool fromBackup, VolumeCache& cache_ref);

		/**
		 * \brief returns interolated voxels density
		 * \param pos position in local space
		 * \param fromBackup take from the backup
		 * \return linearly interplated value of the density
		 */
		float getInterpolatedValue(const vec3& pos, bool fromBackup);

		/**
		 * \brief run through all volume cells
		 * \param fn callback/lambda, VolumeCell pointer passed there
		 * \param multithreaded use multi-threading
		 */
		void scanCells(const std::function<void(VolumeCell* vc)>& fn, bool multithreaded = false);

		/**
		 * \brief run through all volume cells
		 * \param fn callback/lambda, VolumeCellAttrib pointer passed there
		 * \param multithreaded use multi-threading
		 */
		void scanCells(const std::function<void(VolumeCellAttrib* vc)>& fn, bool multithreaded = false);

		/**
		 * \brief run through all volume cells
		 * \param fn callback/lambda, VolumeCellAttrib pointer and space placement passed there
		 * \param multithreaded use multi-threading
		 */
		void scanCells(const std::function<void(VolumeCellAttrib* vc, const tri_DWORD& T)>& fn, bool multithreaded = false);

		/**
		 * \brief run through all volume cells
		 * \param fn callback/lambda, VolumeCell pointer and space placement passed there
		 * \param multithreaded use multi-threading
		 */
		void scanCells(const std::function<void(VolumeCell* vc, const tri_DWORD& T)>& fn, bool multithreaded = false);

		/**
		 * \brief run through all triangles
		 * \param fn the callback/lambda
		 * \param multithreaded use multi-threading
		 */
		void scanTriangles(const std::function<void(const MCVertex& v1, const MCVertex& v2, const MCVertex& v3)>& fn, bool multithreaded = false);

		/**
		 * \brief run through all triangles
		 * \param fn the callback/lambda
		 * \param multithreaded use multi-threading
		 */
		void scanTriangles(const std::function<void(const Vector3D& v1, const Vector3D& v2, const Vector3D& v3)>& fn, bool multithreaded = false);

		/**
		 * \brief get the volume triangles count
		 * \return triangles count
		 */
		int getPolycount();

		/**
		 * \brief get the volume of this object in world coordinates
		 * \return volume
		 */
		float getVolume();

		/**
		 * \brief reg the square of this object in world coordinates
		 * \return square
		 */
		float getSquare();

		/**
		 * \brief Calculate the Axis - Aligned Bound Box of the object in local space
		 * \return the boundary as comms::cBounds
		 */
		boundbox calcLocalSpaceAABB();

		/**
		 * \brief Calculate the Axis - Aligned Bound Box of the object in world space
		 * \return the boundary as comms::cBounds
		 */
		boundbox calcWorldSpaceAABB();

		/**
		 * \brief returns the low-level object (VoxTreeBranch) for all low-level operations
		 * \return the VoxTreeBranch* pointer
		 */
		VoxTreeBranch* tree();

		/**
		 * \brief returns the low-level object (VolumeObject) for all low-level operations
		 * \return the VolumeObject* pointer
		 */
		VolumeObject* vo();

		/**
		 * \brief get the cell by cell coordinates, each cell is 8*8*8
		 * \param cx cell x
		 * \param cy cell y
		 * \param cz cell z
		 * \param create pass true if you want to create the cell if it does not exist
		 * \param backup drop the cell to backup (if not already dropped)
		 * \return the pointer to the VolumeCell
		 */
		VolumeCell* cell(int cx, int cy, int cz, bool create, bool backup);

		/**
		 * \brief get the cell attributes by cell coordinates, each cell is 8*8*8, generally it is kept as VolumeCell::Attr
		 * \param cx cell x
		 * \param cy cell y
		 * \param cz cell z
		 * \param create pass true if you want to create the cell if it does not exist
		 * \param backup drop the cell to backup (if not already dropped)
		 * \return the pointer to the VolumeCellAttrib
		 */
		VolumeCellAttrib* attributes(int cx, int cy, int cz, bool create, bool backup);

		/**
		 * \brief mark the cell as dirty. This is required if you
		 * \param cx
		 * \param cy
		 * \param cz
		 */
		void dirty(int cx, int cy, int cz);

		/**
		 * \brief set the volume opacity
		 * \param Opacity the 0..1 opacity value
		 */
		void setOpacity(float Opacity);

		/**
		 * \brief fast voxel-based relax within the sphere with the gradual falloff. It works only in voxel mode.
		 * \param center the center of 
		 * \param Radius the radius of the influence
		 * \param degree the relax degree, < 1
		 */
		void relaxGpu(const vec3& center, float Radius, float degree);

		/**
		 * \brief relax the whole volume, works only for voxels
		 * \param count the count of relax steps
		 */
		void relaxVoxels(int count);

		/**
		 * \brief relax the object in surface mode
		 * \param degree the degree of smoothing, it may be >1 for the stronger relax
		 * \param tangent use tangent relax
		 * \param keep_sharp_boolean_edges keep the sharp edges appeared due to bolean operations
		 */
		void relaxSurface(float degree, bool tangent = false, bool keep_sharp_boolean_edges = false);

		/**
		 * \brief relax the open edges of the mesh, it is applicable only to the surface mode
		 * \param nTimes amount of iterations
		 */
		void relaxOpenEdges(int nTimes);

		/**
		 * \brief Get the Volume placement in the scene
		 * \return the SceneElement
		 */
		SceneElement inScene();

		/**
		 * \brief Clear and pass to the Undo queue
		 */
		void clear();

		/**
		 * \brief Clear quickly, without affecting the Undo queue
		 */
		void clearNoUndo();

		/**
		 * \brief set the shader for the Volume
		 * \param shaderName the shader name as it is shown in the shader's hint
		 */
		void assignShader(const char* shaderName);

		void setBoolShaderProperty(const char* property, bool value);
		void setFloatShaderProperty(const char* property, float value);
		void setColorShaderProperty(const char* property, DWORD value);

		/**
		 * \brief Remove all faces where all the function returns the value < 0 for all the vertices over the face 
		 * \param weight the function, should return < 0 where you need to remove the whole face, > 0 where you need to keep the face.
		 * Coordinate is in global space.
		 */
		void removeFacesByWeight(std::function<float(const vec3&)> weight);

		/**
		 * \brief Close the holes
		 * \param maxSize max hole size (edges over the primeter)
		 */
		void closeHoles(int maxSize);

		/**
		 * \brief check if molding allowed
		 * \return true if the molding license available
		 */
		static bool checkIfMoldingLicenseAvailable();

		/**
		 * \brief set the parameters for the molding
		 * \param direction the molding direction
		 * \param tapering_angle the tapering angle in degrees
		 * \param undercuts_density the additional density for the undercuts
		 * \param decimation_limit_millions decimate the final shape if it has triangles count more than this value
		 * \param perform_subtraction set false if no need to subtract the molding from the molding shapes
		 */
		static void setMoldingParams(const vec3& direction, float tapering_angle = 0, float undercuts_density = 1.0f, float decimation_limit_millions = 10, bool perform_subtraction = true);

		/**
		 * \brief remove undercuts for the current volume
		 */
		void removeUndercuts();

		/**
		 * \brief perform the bas-relief for the current volume
		 * \param start_point the cut point
		 */
		void basRelief(const vec3& start_point = vec3::Zero);

		/**
		 * \brief set the molding bound box to be automatic
		 */
		static void setAutomaticMoldingBox();

		/**
		 * \brief set the molding bound box to be user-defined, not automatic
		 * \param width the width of the box
		 * \param length the length of the box
		 * \param thickness the thickness of the box
		 */
		static void setMoldingBox(float width, float length, float thickness);

		/**
		 * \brief set the molding border around the parting line to fade to the plane, if it is zero, the final shape will not fade to plane
		 * \param width the width in mm or other default units
		 */
		static void setMoldingBorder(float width = 0);

		/**
		 * \brief generate the automatic molding curves
		 */
		void generateMoldingCurves();

		/**
		 * \brief perform the automatic molding
		 */
		void automaticMolding();

		/**
		 * \brief perform the curve-based mold
		 */
		void curveBasedMolding();

		/**
		 * \brief subtract the current undercutted object from the preliminary generated molding shapes
		 */
		void subtractWithoutUndecuts();

		/**
		 * \brief generate the figure that fills the gap between the molding shapes	
		 * \return the generated scene element reference
		 */
		SceneElement generateMoldingTest();

		/**
		 * \brief find the top molding shape (that was previously generated)
		 * \return the top shape reference
		 */
		SceneElement findMoldingTop();

		/**
		 * \brief find the bottom molding shape (that was previously generated)
		 * \return the bottom shape reference
		 */
		SceneElement findMoldingBottom();

		/**
		 * \brief find the test molding test shape (that was previously generated)
		 * \return the test shape reference
		 */
		SceneElement findMoldingTest();

		/**
		 * \brief remove all molding intermediate shapes, tests, etc.
		 */
		void removeMoldingShapes();

		/**
		 * \brief Apply the live booleans over the sculpt mesh, it is available for voxels only
		 * \param operation 0 - stop live booleans, 1 - subtract from the parent, 2 - intersect, 3 - union
		 */
		void assignLiveBooleans(int operation);

		/**
		 * \brief collapse the boolean tree, it is available for this volume
		 */
		void collapseBollTree();
	};

	inline coat::VolumeCache::VolumeCache() {
		pCell = nullptr;
		CellID.V1 = CellID.V2 = CellID.V3 = 100000;
	}

	class APICALL settings {
	public:
		/**
		 * \brief returns true if the value in settings exists
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \return true if identifier exists
		 */
		static bool valueExists(const char* ID);

		/**
		 * \brief get the boolen value from the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \return the boolean value, false if not exists or casting impossible
		 */
		static bool getBool(const char* ID);

		/**
		 * \brief get the string value from the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \return the string value, empty if not exists
		 */
		static std::string getString(const char* ID);

		/**
		 * \brief get the float value from the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \return the float value, 0 if not exists or casting impossible
		 */
		static float getFloat(const char* ID);

		/**
		 * \brief get the integer value from the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \return the integer value, 0 if not exists or casting impossible
		 */
		static int getInt(const char* ID);

		/**
		 * \brief set the boolean value to the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \param value the value to set
		 * \return true if the value was set successfully
		 */
		static bool setBool(const char* ID, bool value);

		/**
		 * \brief set the string value to the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \param value the value to set
		 * \return true if the value was set successfully
		 */
		static bool setString(const char* ID, const char* value);

		/**
		 * \brief set the float value to the settings
		 * \param ID the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \param value the value to set
		 * \return true if the value was set successfully
		 */
		static bool setFloat(const char* ID, float value);

		/**
		 * \brief set the integer value to the settings
		 * \param the identifier or English text of the option, take identifier from the UI as usual (RMB + MMB)
		 * \param value the value to set
		 * \return true if the value was set successfully
		 */
		static bool setInt(const char* ID, int value);

		/**
		 * \brief save all changed settings
		 */
		static void saveSettings();

		/**
		 * \brief reset all settings to default values, application will restart
		 * \param ResetGeneralSettings reset general settings
		 * \param ResetHiddenSet reset the hidden UI elements list
		 * \param ResetHotkeys reset the hotkeys
		 * \param RestNavigation reset the navigation settings
		 * \param ResetPresets reset the presets
		 * \param ResetTheme reset the theme
		 * \param ResetWindows reset the floating windows placement
		 */
		static void resetSettings(bool ResetGeneralSettings = true, bool ResetHiddenSet = true, bool ResetHotkeys = true, bool RestNavigation = true, bool ResetPresets = true, bool ResetTheme = true, bool ResetWindows = true);

		/**
		 * \brief get the list of all available settings
		 * \return the pairs of strings, first - option identifier, second - the value, pay attention boolean values are "true" and "false" (like in c++)
		 */
		static std::vector<std::string> listAllSettings();

		/**
		 * \brief triger some action in settings
		 * \param button_name the button name, look the identifier in the listAllSettings() output
		 */
		static void pressButton(const char* button_name);
	};

	/**
	 * \brief referes the roots of the scene graph
	 */
	class APICALL Scene {
	public:
		/**
		 * \brief clear the whole scene
		 * \param askUser set true to ask user for unsaved changes
		 */
		static void clearScene(bool askUser = false);
		/**
		 * \brief returns the current sculpt object
		 * \return current object reference
		 */
		static SceneElement current();

		/**
		 * \brief get the root of all sculpt objects
		 * \return the root reference
		 */
		static SceneElement sculptRoot();

		/**
		 * \brief get the root of all curves
		 * \return the root reference
		 */
		static SceneElement curvesRoot();

		/**
		 * \brief get the Layer ID by name, add the layer if not exists
		 * \param name layer name
		 * \param addIfNotExists set true to add layer if it does not exist
		 * \return layer identifier
		 */
		static int getLayer(const char* name, bool addIfNotExists = true);

		/**
		 * \brief get the layer name
		 * \param LayerID the layer identifier
		 * \return the layer name
		 */
		static const char* getLayerName(int LayerID);

		/**
		 * \brief set the layer name
		 * \param LayerID the layer identifier
		 * \param name the new name
		 */
		static void setLayerName(int LayerID, const char* name);

		/**
		 * \brief get the layer blending mode
		 * \param LayerID the layer identifier
		 * \return the index of blending mode as it is ordered in the Layers UI
		 */
		static int getLayerBlending(int LayerID);

		/**
		 * \brief set the layer blending mode
		 * \param LayerID the layer identifier
		 * \param mode the index of blending mode as it is ordered in the Layers UI
		 */
		static void setLayerBlending(int LayerID, int mode);

		/**
		 * \brief get current layer identifier
		 * \return the current layer identifier
		 */
		static int getCurrentLayer();

		/**
		 * \brief set the current layer
		 * \param LayerID the layer identifier
		 */
		static void setCurrentLayer(int LayerID);

		/**
		 * \brief merge all visible layers
		 */
		static void mergeVisibleLayers();

		/**
		 * \brief merge the layer down
		 * \param LayerID the layer identifier
		 */
		static void mergeLayerDown(int LayerID);

		/**
		 * \brief apply layer blending
		 * \param LayerID the layer identifier
		 */
		static void applyLayerBlending(int LayerID);
		
		/**
		 * \brief refresh the layer appearance in scene
		 * \param LayerID the layer identifier
		 */
		static void invalidateLayer(int LayerID);
		
		/**
		 * \brief activate the layer
		 * \param LayerID the layer identifier
		*/
		static void setActiveLayer(int LayerID);
		
		/**
		 * \brief remove the layer
		 * \param LayerID the layer identifier
		 */
		static void removeLayer(int LayerID);

		/**
		 * \brief Check if the layer is empty
		 * \param layerID the layer identifier
		 * \return true if the layer is empty
		 */
		static bool layerIsEmpty(int layerID);

		/**
		 * \brief remove all unused layers
		 */
		static void removeEmptyLayers();

		/**
		 * \brief return the layer visibility
		 */
		static bool layerVisible(int LayerID);

		/**
		 * \brief set the layer visibility
		 * \param Visible the visibility
		 */
		static void setLayerVisibility(int LayerID, bool Visible);

		/**
		 * \brief set the layer opacity
		 * \param LayerID the layer identifier
		 * \param Opacity the color opacity
		 */
		static void setLayerColorOpacity(int LayerID, float Opacity);

		/**
		 * \brief set the layer depth opacity
		 * \param LayerID the layer identifier
		 * \param Opacity the depth opacity
		 */
		static void setLayerDepthOpacity(int LayerID, float Opacity);

		/**
		 * \brief set the layer metalness opacity
		 * \param LayerID the layer identifier
		 * \param Opacity the metalness opacity
		 */
		static void setLayerMetalnessOpacity(int LayerID, float Opacity);

		/**
		 * \brief set the layer gloss/roughness opacity
		 * \param LayerID the layer identifier
		 * \param Opacity the gloss/roughness opacity
		 */
		static void setLayerGlossOpacity(int LayerID, float Opacity);

		/**
		 * \brief assign the mask to the layer if it is not assigned before
		 * \param LayerID the layer identifier to assign the mask
		 * \return the mask identifier
		 */
		static int assignLayerMask(int LayerID);

		/**
		 * \brief remove the layer mask
		 * \param LayerID the layer identifier
		 */
		static void removeLayerMask(int LayerID);

		/**
		 * \brief If the layer has the mask attached, the mask will be extracted as a new layer and the masking disabled
		 * \param LayerID the layer identifier
		 */
		static void extractMaskAsLayer(int LayerID);

		/**
		 * \brief set the MaskLayerID to be used as mask for the LayerID. The MaskLayerID will disappear among the layers list
		 * \param LayerID the layer to be masked
		 * \param MaskLayerID the mask layer
		 */
		static void setMaskForTheLayer(int LayerID, int MaskLayerID);

		/**
		 * \brief enable or disable the layer mask
		 * \param LayerID the layer identifier
		 * \param enable true to enable, false to disable
		 */
		static void enableLayerMask(int LayerID, bool enable);

		/**
		 * \brief check if the mask is enabled for the layer
		 * \param LayerID the layer identifier
		 * \return true if masking is enabled and assigned, false if disabled or not assigned
		 */
		static bool isLayerMaskEnabled(int LayerID);

		/**
		 * \brief invert the layer mask (if assigned)
		 * \param LayerID the layer identifier
		 */
		static void invertLayerMask(int LayerID);

		/**
		 * \brief get the mask identifier assigned to the layer
		 * \param LayerID the layer identifier
		 * \return the mask layer identifier
		 */
		static int getLayerMaskLayer(int LayerID);

		/**
		 * \brief disable the mask for the layer
		 * \param LayerID the layer identifier
		 */
		static void disableLayerMask(int LayerID);

		/**
		 * \brief enable the mask for the layer
		 * \param LayerID the layer identifier
		 */
		static void enableLayerMask(int LayerID);

		/**
		 * \brief check if the mask is enabled for the layer
		 * \param LayerID the layer identifier
		 * \return true if enabled, false if disabled of not assigned
		 */
		static bool maskEnabled(int LayerID);

		/**
		 * \brief set this layer as clipping layer
		 * \param LayerID the layer identifier
		 */
		static void setClippingLayer(int LayerID);

		/**
		 * \brief disable the clipping layer
		 * \param LayerID the layer identifier
		 */
		static void disableClippingLayer(int LayerID);


		/**
		 * \brief Get the count of paint objects in scene
		 * \return the amount
		 */
		static int PaintObjectsCount();

		/**
		 * \brief Get the count of paint materials
		 * \return the amount
		 */
		static int PaintMaterialCount();

		/**
		 * \brief Get the paint UV-sets (textures) count
		 * \return the amount
		 */
		static int PaintUVSetsCount();

		/**
		 * \brief Remove the paint object
		 * \param idx the index of the object
		 */
		static void RemovePaintObject(int idx);

		/**
		 * \brief Remove the paint material
		 * \param idx the index of the material
		 */
		static void RemovePaintMaterial(int idx);

		/**
		 * \brief Remove the UV-set (texture)
		 * \param idx the index of the UV-set (texture)
		 */
		static void RemoveUVSet(int idx);

		/**
		 * \brief Get the reference to the object name
		 * \param idx index of the object
		 * \return the reference
		 */
		static const char* PaintObjectName(int idx);

		/**
		 * \brief Get the reference to the material mane
		 * \param idx the index of the material
		 * \return the reference
		 */
		static const char* PaintMaterialName(int idx);

		/**
		 * \brief Get the reference to the UV set name
		 * \param idx the index of the UV set
		 * \return the reference
		 */
		static const char* PaintUVSetName(int idx);

		/**
		 * \brief import mesh into scene, it is the same as File->Import->Import mesh for vertex painting/reference ... This is the optimal way to import mesh into the scene
		 * \param filename the filename, if it is empty, the dialog appears
		 * \return the scene element reference
		 */
		static SceneElement importMesh(const char* filename, const mat4& transform = mat4::Identity);

		/**
		 * \brief Scale the whole scene visually but keep the export size 
		 * \param scale the scale, >1 means objects become bigger in scene
		 */
		static void ScaleSceneVisually(float scale);

		/**
		 * \brief Keep the scene visial size in scene, but scale the export size
		 * \param scale the scale, >1 means objects become bigger in export
		 */
		static void ScaleSceneUnits(float scale);

		/**
		 * \brief the length of 1 scene unit when you export the scene
		 * \return the 1 unit of scene length in the exported model
		 */
		static float GetSceneScale();

		/**
		 * \brief get the name of the current scene units
		 * \return the name as string
		 */
		static const char* GetSceneUnits();

		/**
		 * \brief Set the scene units without actual scaling the scene to new units, just name change
		 * \param units the name of new units
		 * \return false if units are not supported
		 */
		static bool setSceneUnits(const char* units);

		/**
		 * \brief get the scene shift value, look the Edit->Scale master->X,Y,Z
		 * \return 
		 */
		static vec3 getSceneShift();

		/**
		 * \brief set the scene shift value, look the Edit->Scale master->X,Y,Z
		 * \param shift the new shift value
		 */
		static void setSceneShift(const vec3& shift);

		/**
		 * \brief Get the list of all available units
		 * \return the list of strings
		 */
		static std::vector<std::string> getAvailableUnits();

		/**
		 * \brief Convert the scene units to the new units, the scene scale will be changed, visual size will be kept
		 * \param destination_unit_name the name of new units
		 * \return false if units are not supported
		 */
		static bool convertSceneUnits(const char* destination_unit_name);

		/**
		 * \brief Unify several previous undo operations into one
		 * \param nStack the amount of operations to unify		 
		 */
		static void stackUndo(int nStack);

		/**
		 * \brief reset the texture (stencil or material) transform
		 * \param type 0 - stencil, 1 - material
		 */
		static void resetTexTransform(int type);

		/**
		 * \brief scale the texture (stencil or material)
		 * \param type 0 - stencil, 1 - material
		 * \param scale the additional scale value
		 */
		static void scaleTex(int type, float scale);

		/**
		 * \brief scale the texture (stencil or material) non-uniformly
		 * \param type 0 - stencil, 1 - material
		 * \param scale the 2d scale value
		 */
		static void scaleTexNonUniform(int type, const vec2& scale);

		/**
		 * \brief rotate the texture (stencil or material)
		 * \param type 0 - stencil, 1 - material
		 * \param angle the angle in degrees
		 */
		static void rotateTex(int type, float angle);

		/**
		 * \brief move the texture (stencil or material)
		 * \param type 0 - stencil, 1 - material
		 * \param offset the offset in 2d (screen plane, pixels)
		 */
		static void moveTex(int type, const vec2& offset);

		/**
		 * \brief flip the texture (stencil or material) horizontally
		 * \param type 0 - stencil, 1 - material
		 */
		static void flipTexX(int type);

		/**
		 * \brief flip the texture (stencil or material) vertically
		 * \param type 0 - stencil, 1 - material
		 */
		static void flipTexY(int type);

		/**
		 * \brief make texture tiled or use single tile
		 * \param type 0 - stencil, 1 - material
		 * \param tiled the tiled state
		 */
		static void setTexTiled(int type, bool tiled);

		/**
		 * \brief sep the pivot for the texture (stencil or material)
		 * \param type 0 - stencil, 1 - material
		 * \param pivot the screen coordinates of the pivot
		 */
		static void setTexPivot(int type, const vec2& pivot);

		/**
		 * \brief get the viewport center in screen coordinates
		 * \return the screen coordinates of the viewport center
		 */
		static vec2 getViewportCenter();
	};

	class APICALL RenderRoom {
	public:
		/**
		 * \brief get to the render room to be able to render
		 */
		static void toRenderRoom();

		/**
		 * \brief if the realtime render enabled the command will restart the rendering from scratch
		 */
		static void restartRendering();

		/**
		 * \brief set the render output width 
		 * \param width the width
		 * \param height the height
		 */
		static void setCustomRenderSize(int width, int height);

		/**
		 * \brief set the render output filename
		 * \param filename the filename
		 */
		static void setRenderResult(const char* filename);

		/**
		 * \brief render to the output file
		 */
		static void renderFrame();

		/**
		 * \brief enable or disable the realtime rendering
		 * \param enable set true to enable
		 */
		static void enableRealtimeRendering(bool enable);

		/**
		 * \brief get the realtime rendering state
		 * \return true if enabled
		 */
		static bool isRealtimeRenderingEnabled();

		/**
		 * \brief set the exposure value for the rendering (in render room)
		 * \param exposure the exposure value, usually 0..1, bigger values allowed as well
		 */
		static void setExposure(float exposure);

		/**
		 * \brief get the exposure value for the rendering (in render room)
		 * \return the exposure value, around (0..1)
		 */
		static float getExposure();

		/**
		 * \brief set the brightness of the environment light (spherical environment)
		 * \param envlight the brightness, usually 1
		 */
		static void setEnvironmentLight(float envlight);

		/**
		 * \brief get the brightness of the environment light (spherical environment)
		 * \return the brightness, usually 1
		 */
		static float getEnvironmentLight();

		/**
		 * \brief set the depth of field (DOF) degree
		 * \param degree the degree of DOF, 0 means no DOF, 1 means full DOF
		 */
		static void setDOFDegree(float degree);

		/**
		 * \brief get the depth of field (DOF) degree
		 * \return the degree of DOF, 0 means no DOF, 1 means full DOF
		 */
		static float getDOFDegree();

		/**
		 * \brief get the amount of additional directional lighte
		 * \return the amount
		 */
		static int getLightsCount();

		/**
		 * \brief add the additional directional light
		 * \return the index of the light for all further operations
		 */
		static int addLight();

		/**
		 * \brief remove the additional directional light
		 * \param idx the index of the light
		 */
		static void removeLight(int idx);

		/**
		 * \brief remove all additional directional lights
		 */
		static void removeAllLights();

		/**
		 * \brief set the direction for the additional light
		 * \param idx the index of the light
		 * \param dir the light direction
		 */
		static void setLightDirection(int idx, const vec3& dir);

		/**
		 * \brief get the direction for the additional light
		 * \param idx the index of the light
		 * \return the light direction
		 */
		static vec3 getLightDirection(int idx);

		/**
		 * \brief set the light scattering for the additional light
		 * \param idx the index of the light
		 * \param scattering the light scattering value
		 */
		static void setLightScattering(int idx, float scattering);

		/**
		 * \brief get the light scattering for the additional light
		 * \param idx the index of the light
		 * \return the light scattering value
		 */
		static float getLightScattering(int idx);

		/**
		 * \brief set the light color for the additional light
		 * \param idx the index of the light
		 * \param color the light color (r,g,b) wintin [0..1] range, if need more intensity, increase the light intensity value
		 */
		static void setLightColor(int idx, const vec3& color = vec3::One);

		/**
		 * \brief get the light color for the additional light
		 * \param idx the index of the light 
		 * \return the light color (r,g,b)
		 */
		static vec3 getLightColor(int idx);

		/**
		 * \brief set the light intensity for the additional light
		 * \param idx the index of the light
		 * \param intensity the light intensity value
		 */
		static void setLightIntensity(int idx, float intensity);

		/**
		 * \brief get the light intensity for the additional light
		 * \param idx the index of the light
		 * \return the light intensity value
		 */
		static float getLightIntensity(int idx);

		/**
		 * \brief set rays per frame for the rendering
		 * \param count the rays per frame count
		 */
		static void setRaysPerFrame(int count);

		/**
		 * \brief get rays per frame for the rendering
		 * \return the rays per frame count
		 */
		static int getRaysPerFrame();

		/**
		 * \brief set the anti-aliasing (AA) rendering state
		 * \param AA true to enable
		 */
		static void setAA(bool AA);

		/**
		 * \brief get the anti-aliasing (AA) rendering state
		 * \return true if enabled
		 */
		static bool getAA();
		
	};

	class APICALL Curve:public SceneElement {
		OneCurveObject* cu;
		bool allocated;
		friend class SceneElement;
		void validate();
	public:
		/**
		 * \brief the default constructor, creates the curve without insertion into the scene, it is good for the temporary curves.
		 */
		Curve();
		~Curve();

		/**
		 * \brief create Curve based on low-level object OneCurveObject
		 * \param ob the OneCurveObject pointer
		 */
		Curve(OneCurveObject* ob);

		/**
		 * \brief create the curve object based on the SceneElement, it is valid only is scene element is curve
		 * \param el the scene element
		 */
		Curve(SceneElement& el);
		Curve& operator = (SceneElement& el);

		/**
		 * \brief get the base points cout in the curve 
		 * \return the points count
		 */
		int pointsCount();

		/**
		 * \brief get the base point pointer
		 * \param idx the index in the points array
		 * \return the pointer to the point if it is in range, nullptr othervice
		 */
		OneSelPoint* point(int idx);

		/**
		 * \brief remove the points out of the curve base points list
		 * \param index the start point index
		 * \param count points count to remove
		 */
		void removePoints(int index, int count);

		/**
		 * \brief get the low-level ObjeCurveObject pointer
		 * \return the OneCurveObject pointer
		 */
		OneCurveObject* curve();

		/**
		 * \brief returns the visual points count. Visual points used to render the curve in the viewport as set of straight lines.
		 * \return the count
		 */
		int renderPointsCount();

		/**
		 * \brief returns the visual point reference 
		 * \param idx the point index
		 * \return the pointer to the point if it is in range, nullptr othervice
		 */
		OneSelPoint* renderPoint(int idx);

		/**
		 * \brief update the visual points if need. Use this function if you cahnge the curve. Change the multiple parameters and then call this function if you need visual points.
		 * Othervice they will be updated automatically later.
		 */
		void updatePoints();

		/**
		 * \brief returns the reference to the closed state of the curve to get or set the value
		 * \return the reference
		 */
		bool& closed();

		/**
		 * \brief add the point to the curve without the direct options the tangents
		 * \param p the point in space
		 * \param normal the normal to the point
		 * \param Radius the point radius
		 */
		void add(const Vector3D& p, const Vector3D& normal, float Radius);

		/**
		 * \brief add the sharp point to the curve
		 * \param p the point in space
		 * \param normal the normal to the point
		 * \param Radius the point radius
		 */
		void addSharp(const Vector3D& p, const Vector3D& normal, float Radius);

		/**
		 * \brief add the smooth B-spline-like point to the curve
		 * \param p the position
		 * \param normal the normal
		 * \param Radius the radius
		 */
		void addSmooth(const Vector3D& p, const Vector3D& normal, float Radius);

		/**
		 * \brief add the point with two independent tangents.
		 * \param p the position
		 * \param normal the normal
		 * \param inTangent input tangent, it is usually approximately directed from the current to the previous point
		 * \param outTangent output tangent, it is usually approximately directed from the current to the next point
		 * \param Radius the radius
		 */
		void addBothTangents(const Vector3D& p, const Vector3D& normal, const Vector3D& inTangent, const Vector3D& outTangent, float Radius);

		/**
		 * \brief add the point with the opposite tangents 
		 * \param p the position
		 * \param normal the normal
		 * \param inOutTangent the tangent, it is usually approximately directed from the current to the next point
		 * \param Radius the radius
		 */
		void addWithTangent(const Vector3D& p, const Vector3D& normal, const Vector3D& inOutTangent, float Radius);

		/**
		 * \brief create the solid tube around the curve using the points radius 
		 * \param mesh this mesh will be created as the result of the operation 
		 * \param hemisphere set true if need the ends of the rode to be hemispheres 
		 */
		void tubeToMesh(Mesh& mesh, bool hemisphere);

		/**
		* \brief get the point of the curve
		* \param idx the point index
		* \return the point as tuple (position, normal, tangent1, tangent2, radius)
		*/
		std::tuple<vec3, vec3, vec3, vec3, float> getPoint(int idx);

		/**
		 * \brief set the point position
		 * \param idx the point index
		 * \param p the position
		 */
		void setPointPosition(int idx, const vec3& p);

		/**
		 * \brief set the point normal
		 * \param idx the point index
		 * \param n the normal
		 */
		void setPointNormal(int idx, const vec3& n);

		/**
		 * \brief set the point tangents
		 * \param idx the point index
		 * \param t1 the first tangent
		 * \param t2 the second tangent
		 */
		void setPointTangents(int idx, const vec3& t1, const vec3& t2);

		/**
		 * \brief set the point radius
		 * \param idx the point index
		 * \param r the radius
		 */
		void setPointRadius(int idx, float r);

		/**
		 * \brief check if the curve is open
		 */
		bool isOpen();

		/**
		 * \brief set the curve to be open
		 */
		void setOpen();

		/**
		* \brief set the curve to be closed
		*/
		void setClosed();

		/**
		 * \brief unselect all curve points
		 */
		void unselectPoints();

		/**
		 * \brief select the curve point
		 * \param idx the point index
		 */
		void selectPoint(int idx);

		/**
		 * \brief Create the curved surface around the curve
		 * \param mesh the resulting mesh
		 * \param thickness the thickness of the object
		 * \param relax_count the relaxation degree
		 * \param details_level the details levels
		 * \param extrusion the additional extrusion
		 */
		void fill(Mesh& mesh, float thickness, float relax_count = 0, float details_level = 1, float extrusion = 0);
	};

	/**
	 * \brief The class intended to place spheres in space and identify if there are spheres around. It is important for
	 * random objects generating and avoiding self-intersection of objects
	 */
	class APICALL SphericalCollision {
		list<vec4> spheres;
		uni_hash<int, tri_int> cells;
		float unit;
	public:
		SphericalCollision();

		/**
		 * \brief create the collision space
		 * \param cellsize the cell size that should be approximately around the average sphere size
		 */
		SphericalCollision(float cellsize);

		~SphericalCollision();

		/**
		 * \brief set the cell size, the cell space should be empty
		 * \param u the cell size that should be approximately around the average sphere size
		 */
		void setUnit(float u);

		/**
		 * \brief remove all spheres
		 */
		void clear();

		/**
		 * \brief add the sphere into the space
		 * \param p the position
		 * \param radius the radius
		 * \return the sphere index, you may refer it later using the spher(index) function
		 */
		int addSphere(const vec3& p, float radius);

		/**
		 * \brief check if sphere intersects other spheres in the space
		 * \param p position
		 * \param radius radius
		 * \return the repelling force, it is zero if no collision happened.
		 */
		vec3 collides(const vec3& p, float radius);

		/**
		 * \brief get the sphere parameters by index
		 * \param idx the sphere index (previously returned by addSphere)
		 * \return the position (xyz) and radius (w) as vec4
		 */
		vec4 sphere(int idx);
	};

	/**
	 * \brief operate over the Coat's ui
	 */
	class APICALL ui {
	public:
		/**
		 * \brief execute some action in UI as if you pressed on some control
		 * \details The ID may be taken from the UI by clicking RMB+MMB, then the ID will appear in the clipboard (look Edit->Prferences->General->Script info type).
		 *	If the element triggers modal dialog, the execution will be paused till the modal dialog will be closed, but the callback will be called each frame in modal dialog,
		 *	so you will be able to control what happens in the modal dialog.
		 * \param id the identifier taken from the UI. It looks like "$SOMETHING_THERE". If you will call the cmd with ID = "$[page_]SOMETHING_THERE" then only current page UI elements will be searched.
		 * Other-vice, the item will be searched globally among the all UI elements.
		 * \param process_in_modal_dialog the callback/lambda that will be called each frame till you are within the modal dialog
		 * \return true if the element found and the operation executed
		 */
		static bool cmd(const char* id, std::function<void()> process_in_modal_dialog = 0);
#ifdef PYBIND11_DEF
		/**
		 * \brief execute some action in UI as if you pressed on some control
		 * \details The ID may be taken from the UI by clicking RMB+MMB, then the ID will appear in the clipboard (look Edit->Prferences->General->Script info type).
		 *	If the element triggers modal dialog, the execution will be paused till the modal dialog will be closed, but the callback will be called each frame in modal dialog,
		 *	so you will be able to control what happens in the modal dialog.
		 * \param id the identifier taken from the UI
		 * \param fn the callback/lambda that will be called each frame till you are within the modal dialog. It looks like\n
		 * \code
		 * def _callback():
		 *     cmd("#id_to_press")
		 *	   ...code...
		 * \endcode
		 * \return True if the element found and the operation executed
		 */
		static bool cmd(const char* id, pybind11::object fn = pybind11::none());
#endif
		/**
		 * \brief wait till the element id will appear in the UI. The element will not be clicked. The max wait time is max_seconds.
		 * \param id The ID we wait to appear
		 * \param max_seconds the max wait time (seconds)
		 * \return true if the element appeared
		 */
		static bool wait(const char* id, float max_seconds);

		/**
		 * \brief Check if the elemnt present in the UI
		 * \param id the identifier
		 * \return true if the element is present
		 */
		static bool presentInUI(const char* id);

		/**
		 * \brief highlight the UI element for a while
		 * \param id the ID of the element
		 * \param milliseconds the time to highlight, milliseconds
		 */
		static void highlight(const char* id, float milliseconds);

		/**
		 * \brief enable or disable the pen channel
		 * \param i the channel: 0 - depth, 1 - color, 3 - gloss, 2 - currently unused
		 * \param enabled true to enable
		 */
		static void enablePenChannel(int i, bool enabled);

		/**
		 * \brief check if the pen channel is enabled
		 * \param i the cannel: 0 - depth, 1 - color, 3 - gloss, 2 - currently unused
		 * \return true if the channel is enabled
		 */
		static bool isEnabledPenChannel(int i);

		/**
		 * \brief Set the value for the the slider (if exists in UI)
		 * \param id the ID of the element
		 * \param value the value to set
		 * \return true if successful
		 */
		static bool setSliderValue(const char* id, float value);

		/**
		 * \brief get the value of the slider
		 * \param id the ID of the element
		 * \return the value
		 */
		static float getSliderValue(const char* id);

		/**
		 * \brief set the edit box value
		 * \param id the ID of the element
		 * \param value the value to set
		 * \return true if the element exists
		 */
		static bool setEditBoxValue(const char* id, const char* value);

		/**
		 * \brief set the edit box value
		 * \param id the ID of the element
		 * \param value the value to set
		 * \return true if the element exists
		 */
		static bool setEditBoxValue(const char* id, int value);

		/**
		 * \brief set the edit box value
		 * \param id the ID of the element
		 * \param value the value to set
		 * \return true if the element exists
		 */
		static bool setEditBoxValue(const char* id, float value);

		/**
		 * \brief get the edit box value
		 * \param id the ID of the element
		 * \param result the string the will get the result
		 * \return true if the element exists
		 */
		static bool getEditBoxValue(const char* id, str& result);
		static const char* getEditBoxValue(const char* id);

		/**
		 * \brief pess ENTER, acts as Apply usually
		 */
		static void apply();

		/**
		 * \brief Set the file for the next file dialog that will be triggered by user.
		 * If you will use coat::ui:cmd(...) to trigger some command that shows the file dialog
		 * this command allows to substitute the filename for that dialog instead of showing the dialog.
		 * This acts only for ONE next dialog.
		 * \param filename the filename to substitute.
		 */
		static void setFileForFileDialog(const char* filename);

		/**
		 * \brief Get the bool field from the checkbox in UI
		 * \param id the element identifier
		 * \return the checkbox value
		 */
		static bool getBoolField(const char* id);

		/**
		 * \brief Set the value for the checkbox in UI
		 * \param id the element identifier
		 * \param value the value to set
		 * \return true if successful and the element exists
		 */
		static bool setBoolValue(const char* id, bool value);

		/**
		 * \brief get the current room name
		 * \return the name
		 */
		static const char* currentRoom();

		/**
		 * \brief check if we are in the specified room
		 * \param name the room name to check
		 * \return true if we are in that room
		 */
		static bool isInRoom(const char* name);

		/**
		 * \brief switch to the room
		 * \param name the room name. Pay attention, you may pass the name or identifier, but name has bigger priory.
		 * \param Force set true to switch even if we are in the tool that corresponds to the destination room
		 */
		static void toRoom(const char* name, bool Force = false);

		/**
		 * \brief returns the rooms count
		 * \return the number
		 */
		static int roomsCount();

		/**
		 * \brief get the room name by index
		 * \param index the room index
		 * \return "" if index outside the range or the room name if the index is valid
		 */
		static const char* roomName(int index);

		/**
		 * \brief get the text identifier of the room
		 * \param index the room index
		 * \return "" if index outside the range or the room identifier if the index is valid
		 */
		static const char* roomID(int index);

		/**
		 * \brief show the class B as the part of the tools params panel
		 * \param B the class pointer
		 */
		static void toolParam(BaseClass* B);

		/**
		 * \brief remove the class from the tools params
		 * \param B the class pointer
		 */
		static void removeToolParam(BaseClass* B = nullptr);

		/**
		 * \brief get the option from preferences
		 * \param id the identifier of english text of the option
		 * \return the value as string
		 */
		static const char* getOption(const char* id);

		/**
		 * \brief set the value to preferences
		 * \param id the value identifier or english text 
		 * \param value 
		 * \return 
		 */
		static bool setOption(const char* id, const char* value);
		static bool setOption(const char* id, bool value);
		static bool setOption(const char* id, float value);

		/**
		 * \brief Hides the "Don't show again dialog" for the current session (not forever)
		 * \param id the identifier, for example "AttachTextureHint", look the currently hidden list as files names in Docs/3DCoat/data/Temp/*.dontshow
		 */
		static void hideDontShowAgainMessage(const char* id);

		/**
		 * \brief Show the floating information message for the some time period
		 * \param infoID the message or message identifier (from language files)
		 * \param milliseconds how ling to display the message
		 */
		static void showInfoMessage(const char* infoID, int milliseconds);

		/**
		 * \brief Insert the scripted command into the main menu
		 * \param Menu One of main menu items, look the list in Documents/3DCoat/UserPrefs/Scripts/ExtraMenuItems/menu_sections.txt
		 * \param ID_in_menu the ID of the command in the menu, it is the english text or the identifier of the command
		 * \param script_path the full or relative path to the script file, if relative, it is relative to the 3DCoat root folder
		 * If it comes without path, it is assumed to be in same folder as the script that calls this function. If this parameter is empty,
		 * this script will be called.
		 */
		static void insertInMenu(const char* Menu, const char* ID_in_menu, const char* script_path);

		/**
		 * \brief Insert the script-based tool into the toolset
		 * \param roomID the room identifier, same as folders names in Documents/3DCoat/UserPrefs/Rooms/CustomRooms/
		 * \param section the section name. This string may be empty to add beyond sections (anyway, at the end) or in any existing section.
		 * To get section name, pres RMB+MMB over the section name in the toolset. You will get something like "*Adjust" in the clipboard.
		 * The "Adjust" in this case is the section name.
		 * \param toolID the tool identifier, how it will appear in UI. You may provide the text for the identifier using the addTranslation(...)
		 * Also, if there is image in the data/Textures/icons64/ named as toolID.png, it will be used as the icon for the tool.
		 * \param script_path the full or relative path to the script file, if relative, it is relative to the 3DCoat root folder
		 * If it comes without path, it is assumed to be in same folder as the script that calls this function. If this parameter is empty,
		 * this script will be called.
		 */
		static void insertInToolset(const char* roomID, const char* section, const char* toolID, const char* script_path = "");

		/**
		 * \brief remove the command from the menu
		 * \param ID_in_menu the ID of the command in the menu, it is the english text or the identifier of the command
		 */
		static void removeCommandFromMenu(const char* ID_in_menu);

		/**
		 * \brief Check if the command inserted somewhere into the menu
		 * \param ID_in_menu the ID of the command in the menu, look the list in C:/Users\andre\OneDrive\Documents\3DCoat/UserPrefs\Scripts\ExtraMenuItems\menu_sections.txt it is the english text or the identifier of the command
		 * \return true if the command is inserted
		 */
		static bool checkIfMenuItemInserted(const char* ID_in_menu);

#ifdef PYBIND11_DEF
		/**
		 * \brief Add the extension (new tool) into the room. Look the \ref GeneratorExample.py 
		 * \param roomID  roomID the room identifier, same as folders names in Documents/3DCoat/UserPrefs/Rooms/CustomRooms/
		 * \param section section the section name. This string may be empty to add beyond sections (anyway, at the end) or in any existing section.
		 * To get section name, pres RMB+MMB over the section name in the toolset. You will get something like "*Adjust" in the clipboard.
		 * The "Adjust" in this case is the section name.
		 * \param obj the object that contains the extension. Look the \ref GeneratorExample.py
		 */
		static void addExtension(const char* roomID, const char* section, pybind11::object& obj);
#endif
		/**
		 * \brief Check if extension named as extension_ID is present in the 3DCoat
		 * \param extension_ID the identifier of the extension
		 * \return True if the extension installed
		 */
		static bool checkIfExtensionPresent(const char* extension_ID);

		/**
		 * \brief Add the translation for the text identifier
		 * \param id the identifier
		 * \param text the translation
		 */
		static void addTranslation(const char* id, const char* text);

		/**
		 * \brief Get the translation for the text identifier
		 * \param id the text identifier
		 * \return the translation or the id
		 */
		static const std::string getIdTranslation(const char* id);

		/**
		 * \brief Get the current language file name (without the XML extension)
		 * \return the language file name (without the XML extension)
		 */
		static const std::string getCurrentLanguage();

		/**
		 * \brief Switch the layout to the language
		 * \param language the language identifier, actually it is the file name (withot the XML extension) in the data/Languages/ folder
		 */
		static void switchToLanguage(const char* language);

		/**
		 * \brief returns the scale in comparison to the smallest UI theme
		 * \return the scale factor > 1
		 */
		static float scale();

		/**
		 * \brief input text under the mouse position
		 * \param text the initial text value
		 * \param min_length the minimal width of the input field, if zero passed the width taken from the parent control (if exists)
		 * \return the changed text (if the user pressed OK) or the initial text other vice.
		 */
		static std::string inputString(const char* text, int min_length = 0);

		static void nomodalInput(str* text, std::function<void()> on_change = nullptr, int dx = 0, int dy = 0);

		static bool nomodalInputActive();

		/**
		 * \brief input the integer value under the mouse position
		 * \param initial_value the initial integer value
		 * \return the changed value (if the user pressed OK) or the initial value other vice.
		 */
		static int inputInt(int initial_value);

		/**
		 * \brief inputh the float value under the mouse position
		 * \param initial_value the initial float value
		 * \return the changed value (if the user pressed OK) or the initial value other vice.
		 */
		static float inputFloat(float initial_value);
	};

	class APICALL Camera
	{
	public:
		/**
		 * \brief align the camera along the view
		 * \param destination_dir the view direction
		 */
		static void rotateToGradually(const vec3& destination_dir);

		/**
		 * \brief get the forward direction
		 * \return the direction
		 */
		static vec3 getForward();

		/**
		 * \brief get the camera up direction
		 * \return the direction
		 */
		static vec3 getUp();

		/**
		 * \brief get the camera right direction
		 * \return the direction
		 */
		static vec3 getRight();

		/**
		 * \brief return true if the camera is in the ortho mode
		 * \return the ortho mode state
		 */
		static bool isOrtho();

		/**
		 * \brief switch the camera to the ortho or perspective mode
		 * \param ortho set true if need ortho mode, false if need perspective mode
		 */
		static void setOrtho(bool ortho);

		/**
		 * \brief get the camera pivot position
		 * \return the position
		 */
		static vec3 getPivot();

		/**
		 * \brief set the camera pivot position
		 * \param pivot the pivot position
		 */
		static void setPivot(const vec3& pivot);

		/**
		 * \brief get the camera position
		 * \return the camera position
		 */
		static vec3 getPosition();

		/**
		 * \brief convert the world position to the screen position
		 * \param world_pos the world position
		 * \return the screen position
		 */
		static vec3 getWorldToScreenSpace(const vec3& world_pos);

		/**
		 * \brief convert the screen position to the world position
		 * \param screen_pos the screen position (pass z that you got using getWorldToScreenSpace)
		 * \return the world position
		 */
		static vec3 getScreenToWorldSpace(const vec3& screen_pos);

		/**
		 * \brief 
		 * \param  
		 * \param  
		 * \param  
		 * \param  
		 * \param  
		 */
		static void setCamera(const vec3& position, const vec3& lookAt, float fovY, const vec3& up = vec3::Zero);
	};

	/**
	 * \brief the rich dialog. You may customize it, show your custom parameters and custom buttons.
	 */
	class APICALL dialog {
#ifndef DOXYGEN_SHOULD_SKIP_THIS
		bool _modal;
		bool _topright;
		str _buttons;
		std::function<void()> _process;
		std::function<void(int)> _press;
		int _width;
		int _icon;
		bool _undoWorks;
		bool _dontshowagain;
		BaseClass* _bc;
		str _text;
		str _caption;
		bool _transparentbg;
		BaseWidget* _widget;
#endif //DOXYGEN_SHOULD_SKIP_THIS
	public:
		dialog();
		/**
		 * \brief pass the header text of the dialog
		 * \param id the text or text identifier that will be used to take the text from the language file. You may press F9 to localize it in UI.
		 * \return the reference to pass multiple options at chain.
		 */
		dialog& text(const char* id);

		/**
		 * \brief pass the caption of the dialog
		 * \param id id the caption or caption identifier that will be used to take the text from the language file. You may press F9 to localize it in UI.
		 * \return the chain ref
		 */
		dialog& caption(const char* id);

		/**
		 * \brief change the default width
		 * \param w the width will be scaled in correspondence with the font size, so you may pass absolute values like 500
		 * \return itself
		 */
		dialog& width(int w);

		/**
		 * \brief dialog will be modal. Generally, it is modal by default. Execution will be paused at show() till the user will press any dialog button.
		 * \return itself
		 */
		dialog& modal();

		/**
		 * \brief dialog will be no modal. Execution will continue after you will call the show()
		 * \return
		 */
		dialog& noModal();

		/**
		 * \brief pass the list of buttons for the dialog
		 * \param list list of buttons. |, .+; may be used as separators between identifiers
		 * \return itself
		 */
		dialog& buttons(const char* list);

		/**
		 * \brief place the dialog at the top-right position of the viewport
		 * \return itself
		 */
		dialog& topRight();

		/**
		 * \brief add Ok button
		 * \return itself
		 */
		dialog& ok();

		/**
		 * \brief add Cancel button
		 * \return itself
		 */
		dialog& cancel();

		/**
		 * \brief add Yes button
		 * \return itself
		 */
		dialog& yes();

		/**
		 * \brief add No button
		 * \return itself
		 */
		dialog& no();

		/**
		 * \brief add Warning icon
		 * \return itself
		 */
		dialog& warn();

		/**
		 * \brief add Question icon
		 * \return itself
		 */
		dialog& question();

		/**
		 * \brief allow undo (CTR-Z) act even in modal dialog
		 * \return itself
		 */
		dialog& undoWorks();

		/**
		 * \brief the background will not be faded
		 * \return itself
		 */
		dialog& transparentBackground();

		/**
		 * \brief returns the reference to the global property - fade modal dialogs background (true) or not (false)
		 * \return the property reference
		 */
		static bool& fadeDialogsBackground();

		/**
		 * \brief show the checkbox "Don't show again". If user checks if the dialog will net be shown next time and show() will return 0 immediately.
		 * \return itself
		 */
		dialog& dontShowAgainCheckbox();

		/**
		 * \brief The important core feature. BaseClass allows to create the custom controls in the dialog. Look the dialog example to understand how to use it.
		 * \param params the pointer to the class derived from the BaseClass
		 * \return itself
		 */
		dialog& params(BaseClass* params);
#ifdef PYBIND11_DEF
		/**
		 * \brief The important core feature. Pass the object to display object parameters in UI. Look the dialog example to understand how to use it.
		 * \param params the class reference
		 * \return itself
		 */
		dialog& params(pybind11::object params);

		/**
		 * \brief pass the function/lambda that will be called each frame.
		 * \param callback the callback/lambda called each frame within the dialog
		 * \return itself
		 */
		dialog& process(pybind11::object callback);
#endif
		/**
		 * \brief pass the function/lambda that will be called each frame. But there is better way - override the ProcessInEditor for the BaseClass derived class and handle the processing there.
		 * \param process the callback/lambda
		 * \return itself
		 */
		dialog& process(std::function<void()> process);

		/**
		 * \brief pass the function/lambda that will be called when the button will be pressed. The button index (starts from 1) will be passed to the function
		 * \param press the callback/lambda
		 * \return itself
		 */
		dialog& onPress(std::function<void(int)> press);
#ifdef PYBIND11_DEF/**
		 * \brief pass the function/lambda that will be called when the button will be pressed. The button index (starts from 1) will be passed to the function
		 * The function looks lite:
		 * \code 
		 * def press(index):
		 *     if index == 1:
		 *         print("Ok pressed, usually it is left button, so it las index 1, indices start from 1 and grow to the right")
		 * \endcode
		 * \param press the callback/lambda
		 * \return itself
		 */
		dialog& onPress(pybind11::object press);
#endif

		/**
		 * \brief Show the dialog. This is usually the last command in the chain.
		 * \return the button index. First button in the list has index 1
		 */
		int show();

		dialog& widget(BaseWidget* w);
	};

	/**
	 * \brief this class represents different resources accessible in UI - alphas, strips, materials, models, etc
	 */
	class APICALL resource {
		ItemsFolder* folder;
	public:
		/**
		 * \brief create the reference to some resource type to access it and operate with it
		 * \param id the resource identifier (string), possible cases:\n
		 * "Alphas", "Strips", "SmartMaterials", "Stencils", "Models/SculptModels", "Models/SplineProfiles",
		 * "Models/RetopoModels", "Models/Joints", "Presets", "Panoramas", "Shaders/PbrShaders", "Stencils" 
		 */
		resource(const char* id);

		/**
		 * \brief list all available resources types
		 * \return the list of resource types (any type may be passed to the resource constructor)
		 */
		static std::vector<std::string> listAllResourcesTypes();

		/**
		 * \brief list folders of the resource type referred by this object
		 * \return the list of folders (short names without the full path)
		 */
		std::vector<std::string> listFolders();

		/**
		 * \brief get the current folder short name
		 * \return the name of current folder of the resource type referred by this object
		 */
		std::string currentFolder();

		/**
		 * \brief full path (relative to the 3DCoat's documents) to the current folder files 
		 * \return the path
		 */
		std::string currentFolderFullPath();

		/**
		 * \brief the root path (relative to the 3DCoat's documents) to the resource type referred by this object
		 * \return the path
		 */
		std::string rootPath();

		/**
		 * \brief get the list of supported extensions for the resource type referred by this object
		 * \return the list of strings with extensions
		 */
		std::vector<std::string> supportedExtensions();

		/**
		 * \brief set the current folder (short name without the full path)
		 * \param folder the folder name
		 */
		void setCurrentFolder(const char* folder);

		/**
		 * \brief create the folder and switch there
		 * \param folderName the folder name
		 */
		void createFolder(const char* folderName);

		/**
		 * \brief remove the folder and switch to the root folder if this is the current folder
		 * \param folderName the folder name
		 */
		void removeFolder(const char* folderName);

		/**
		 * \brief get the list of all items in the current folder
		 * \return the list of items (usually long names with the relative path)
		 */
		std::vector<std::string> listCurrentFolderItems();

		/**
		 * \brief add the item to the current folder
		 * \param itemPath the path to the item (full or relative to the 3DCoat's documents)
		 */
		void addItem(const char* itemPath);

		/**
		 * \brief remove the item from the current folder
		 * \param itemName the item name as it is returned by listCurrentFolderItems()
		 */
		void removeItem(const char* itemName);

		/**
		 * \brief select/activate the item in the current folder
		 * \param itemName the item name as it is returned by listCurrentFolderItems()
		 */
		void selectItem(const char* itemName);

		/**
		 * \brief move the item to another folder
		 * \param itemName the item name as it is returned by listCurrentFolderItems()
		 * \param destFolderName the short name of the destination folder
		 */
		void moveItemToFolder(const char* itemName, const char* destFolderName);

		/**
		 * \brief returns the current item name (if possible)
		 * \return the string, current item name
		 */
		std::string getCurrentItem();
	};

	/**
	 * \brief General I/O access
	 */
	class APICALL io {
	public:
		/**
		 * \brief the 3DCoat installation path
		 * \return the path
		 */
		static const char* installPath();

		/**
		 * \brief the 3DCoat data path
		 * \return the path
		 */
		static const char* dataPath();

		/**
		 * \brief convert the relative path to the path in documents, if the path is absolute, just return the original path
		 * \param path the relative or absolute path
		 * \return the absolute path in user documents
		 */
		static const char* documents(const char* path);

		/**
		 * \brief check if file exists
		 * \param path the path may be full or relative. If it is relative, the documents will be
		 * checked first, the the install folder will be checked for file.
		 * \return true if the file exists
		 */
		static bool fileExists(const char* path);

		/**
		 * \brief copy the file from src to dest. If the src or dest is relative, it is relative to the documents folder. This function works correctly with relative paths, so it is recommended over the standard copy files routine.
		 * \param src the source filename
		 * \param dest the destination filename
		 */
		static void copyFile(const char* src, const char* dest);

		/**
		 * \brief copy the whole folder from src to dest. If the src or dest is relative, it is relative to the documents folder. This function works correctly with relative paths, so it is recommended over the standard copy folder routine.
		 * \param src the source folder
		 * \param dest the destination folder
		 */
		static void copyFolder(const char* src, const char* dest);

		/**
		 * \brief remove the file. If the filename is relative, it is relative to the documents folder. If the path is in install folder, the corresponding file in documents will be removed.
		 * Files in the install folder can't be removed.
		 * \param filename the file path
		 */
		static void removeFile(const char* filename);

		/**
		 * \brief remove the folder. If the folder is relative, it is relative to the documents folder. If the path is in install folder, the corresponding folder in documents will be removed.
		 * \param folder the path to the folder
		 */
		static void removeFolder(const char* folder);

		/**
		 * \brief convert the relative path to full path in documents folder. If the path is full and placed
		 * in the install folder, it will be converted to path in documents.
		 * \param path the path
		 * \return the fill path in documents
		 */
		static std::string toFullPathInDataFolder(const char* path);
		static void toFullPathInDataFolder(str* path);

		/**
		 * \brief convert the relative path to the full path in the install folder.
		 * If the path is full, it remains untouched.
		 * \param path the relative path
		 * \return the full path in the install folder
		 */
		static std::string toFullPathInInstallFolder(const char* path);
		static void toFullPathInInstallFolder(str* path);

		/**
		 * \brief If the path is relative or points into some file in the install folder, it will be converted to the path in documents folder.
		 * \param path the path (full or relative)
		 * \return the write-able path
		 */
		static std::string convertToWritablePath(const char* path);
		static void convertToWritablePath(str* path);

		/**
		 * \brief If the path is relative or points into some file in the install folder, it will be converted to the path in documents folder.
		 * If the does not exist in the documents folder, but exists in the install folder, the resulting path will be in the install folder.
		 * \param path the path (full or relative)
		 * \return the path
		 */
		static std::string convertToWritablePathIfFileExists(const char* path);
		static void convertToWritablePathIfFileExists(str* path);

		/**
		 * \brief get the file extension (without .)
		 * \param filepath the file path - full or relative
		 * \return the extension
		 */
		static std::string getExtension(const char* filepath);

		/**
		 * \brief get the file name from the path
		 * \param filepath the full or relative path
		 * \return the filename without the path
		 */
		static std::string getFileName(const char* filepath);

		/**
		 * \brief get the file path without the filename
		 * \param filepath the filepath
		 * \return the path that always ends with '/'
		 */
		static std::string getFilePath(const char* filepath);

		/**
		 * \brief remove the file extension from the filename
		 * \param filepath the file name
		 * \return the filename without extension
		 */
		static std::string getFileNameWithoutExtension(const char* filepath);

		/**
		 * \brief read string from file.
		 * \param filename The path. If it is relative, it is relative to the documents folder.
		 * If there is no file, it will be taken from the install folder.
		 * \return true if file read succesful
		 */
		static std::string strFromFile(const char* filename);

		/**
		 * \brief write the string to file
		 * \param text the text to save
		 * \param filename The path. If it is relative, it is relative to the documents folder.
		 */
		static void strToFile(const char* text, const char* filename);

		/**
		 * \brief get the file size
		 * \param filename the filename, relative or full
		 * \return the file size
		 */
		static size_t getFileSize(const char* filename);

		/**
		 * \brief returns the current cursor position
		 * \return the 2d vector
		 */
		static vec2 cursorPos();

		/**
		 * \brief returns the snapped cursor position
		 * \return the 2d vector
		 */
		static vec2 snappedCursorPos();

		/**
		 * \brief get the whole screen rectangle
		 * \return the rectangle, top-left is (0,0)
		 */
		static rect wholeScreen();

		/**
		 * \brief get the work area rectangle
		 * \return the rectangle, top-left is (0,0)
		 */
		static rect workArea();

		/**
		 * \brief Show the progress bar
		 * \param stage the current stage
		 * \param max_stage the maximal stage
		 * \param message the text to display
		 */
		static void progressBar(float stage, float max_stage, const char* message);

		/**
		 * \brief Show the progress bar only in the 3DCoat's window header
		 * \param stage the current stage
		 * \param max_stage the maximal stage
		 * \param message the text to display
		 */
		static void progressBarInWindowHeader(float stage, float max_stage, const char* message);

		/**
		 * \brief Override the 3DCoat's window title for some amount of time
		 * \param text the text to show
		 * \param seconds the seconds to stay in title
		 */
		static void setWindowTitle(const char* text, float seconds);

		/**
		 * \brief perform rendering cycles
		 * \param count amount of cycles
		 */
		static void step(int count = 1);

		/**
		 * \brief execute command. It may be exe file, URL, batch command
		 * \param command the command to execute
		 * \param arguments optional command line arguments 
		 */
		static void exec(const char* command, const char* arguments = nullptr);

		/**
		 * \brief execute and wait till finished, the console output will be returned as string
		 * \param command the command to execute
		 * \param arguments optional arguments
		 * \return the console output of the executed program
		 */
		static const char* execAndWait(const char* command, const char* arguments = nullptr);


		/**
		 * \brief update the .pyi file for the given folder or py file
		 * \param folderOrFile the full or relative path to the folder or py file
		 */
		static void updateCoatPyi(const char* folderOrFile);

		/**
		 * \brief list files in the folder
		 * \param folder the start folder. It may be absolute or relative to 3DCoat documents/install folder.
		 * \param mask the seek mask (wildcards)
		 * \param result the files list
		 * \param recursive set true if recursive
		 */
		static void ListFiles(const char* folder, const char* mask, coat::list<coat::str>& result, bool recursive = true);

		/**
		 * \brief list files in the folder
		 * \param folder the start folder. It may be absolute or relative to 3DCoat documents/install folder.
		 * \param mask the seek mask (wildcards)
		 * \param recursive set true if recursive
		 * \return result the files list
		 */
		static std::vector<std::string> ListFiles(const char* folder, const char* mask, bool recursive = true);

		/**
		 * \brief list folders within the folder, non-recursive, just plain list
		 * \param startFolder the start folder
		 * \param result the resulting list
		 */
		static void ListFolders(const char* startFolder, coat::list<coat::str>& result);

		/**
		 * \brief list folders within the folder, non-recursive, just plain list
		 * \param startFolder the start folder
		 * \return the resulting list
		 */
		static std::vector<std::string> ListFolders(const char* startFolder);

		/**
		 * \brief returns the currently supported mesh export formats
		 * \return the list like "*.obj;*.fbx;..."
		 */
		static const char* supportedImagesFormats();

		/**
		 * \brief returns the list of supported images formats
		 * \return the list like "*.png;*.jpg;..."
		 */
		static const char* supportedMeshesFormats();

		/**
		 * \brief show the file dialog
		 * \param extensions the list of supported extensions like *.txt;*.dat; etc...
		 * \param fileName the resulting filename
		 * \return true if user chosen the file successfully
		 */
		static bool openFileDialog(const char* extensions, str& fileName);
		static std::string openFileDialog(const char* extensions);

		/**
		 * \brief open multiple files dialog
		 * \param extensions the list of supported extensions like *.txt;*.dat; etc...
		 * \param fileNames the resulting filename
		 * \return true if user chosen the file successfully
		 */
		static bool openFilesDialog(const char* extensions, list<str>& fileNames);

		/**
		 * \brief open multiple files dialog
		 * \param extensions the list of supported extensions like *.txt;*.dat; etc...
		 * \return the resulting filenames list
		 */
		static std::vector<std::string> openFilesDialog(const char* extensions);

		/**
		 * \brief show the save file dialog
		 * \param extensions extensions the list of supported extensions like *.txt;*.dat; etc...
		 * \param fileName the resulting filename
		 * \return true if user chosen the file successfully
		 */
		static bool saveFileDialog(const char* extensions, str& fileName);
		static std::string saveFileDialog(const char* extensions);

		/**
		 * \brief returns the current scene filename, empty if the scene was not saved/opened 
		 */
		static const char* currentSceneFilepath();

		/**
		 * \brief install one or multiple python packages
		 * \param requirements the list of packages to install, this is all what you write after "pip install" 
		 */
		static void pipInstall(const char* requirements);
		static void pipUninstall(const char* requirements);

		/**
		 * \brief get the python libraries folder
		 * \return the path
		 */
		static std::string pythonPath();

		/**
		 * \brief Show the python console, clear it and pop up
		 */
		static void showPythonConsole();

		/**
		 * \brief execute python (.py file) or angelscript (c++ like), or CoreAPI (native C++) script
		 * \param path the full or relative path to the script file
		 */
		static void executeScript(const char* path);


		/**
		 * \brief Install all the requirements for the python script execution
		 * \param path_to_requirements_txt te full or relative path to the requirements.txt
		 */
		static void installRequirements(const char* path_to_requirements_txt);
#ifdef PYBIND11_DEF
		/**
		 * \brief Store the object to the file or string as json
		 * \param obj the python object reference
		 * \param filename the filename to save, if empty, the string will be returned
		 * \return the string that contains json data
		 */
		static std::string toJson(const pybind11::object& obj, const char* filename = "");

		/**
		 * \brief Restore the object from the json file
		 * \param obj the object to restore
		 * \param filename the path to the json file, full or relative
		 */
		static void fromJsonFile(pybind11::object& obj, const char* filename);

		/**
		 * \brief Restore the object from the json string
		 * \param obj the object to restore
		 * \param data the json string
		 */
		static void restoreObjectFormJsonString(pybind11::object& obj, const char* data);
#endif

		/**
		 * \brief Create the 3dcpack file from the folder placed in Documents
		 * \param folder the folder to pack, it should be relative to the 3DCoat's Documents folder
		 * \param package_name the package name, the extension is .3dcpack
		 * \param excluded_folders_names the folders names to be excluded from the package, separated by semicolon
		 * \param excluded_extensions the file extensions to be excluded from the package, separated by semicolon
		 */
		static void createRedistributablePackageFromFolder(const char* folder, const char* package_name, const char* excluded_folders_names = "", const char* excluded_extensions = "");

		/**
		 * \brief download the file from the url
		 * \param url the url
		 * \param report_success the callback that gives you the source URL and the filename of the downloaded file 
		 */
		static void download(const char* url, std::function<void(const char*, const char*)> report_success);

		/**
		 * \brief returns the overall download progress
		 * \return the progress in percents
		 */
		static int getDownloadProgress();

		/**
		 * \brief send the post request to the url
		 * \param url the url
		 * \param data the body of the request
		 * \param headers the headers of the request, separated by \n
		 * \param report_success the callback that gives you the response if the request was successful
		 * \param report_error the callback that gives you the response if the request was failed
		 */
		static void post_request(const char* url, const char* data, const char* headers, const std::function<void(const char*)>& report_success = nullptr, const std::function<void(const char*)>& report_error = nullptr);

		/**
		 * \brief send the get request to the url
		 * \param url the url
		 * \param headers the headers of the request, separated by \n
		 * \param report_success the callback that gives you the response if the request was successful
		 * \param report_error the callback that gives you the response if the request was failed
		 */
		static void get_request(const char* url, const char* headers, const std::function<void(const char*)>& report_success = nullptr, const std::function<void(const char*)>& report_error = nullptr);

		/**
		 * \brief list the blender install folders
		 * \return the list of the blender install folders
		 */
		static std::vector<std::string> listBlenderInstallFolders();

		/**
		 * \brief save the screenshot to the file
		 * \param filename the filename
		 * \param x the x coordinate of the screenshot
		 * \param y the y coordinate of the screenshot
		 * \param width the width of the screenshot, if 0 all to the right will be captured
		 * \param height the height of the screenshot, if 0 all to the bottom will be captured
		 */
		static void saveScreenshot(const char* filename, int x = 0, int y = 0, int width = 0, int height = 0);

		static void removeBackground(const char* image1, const char* image2, const char* result);
	};

	class APICALL utils	{
	public:
		/**
		 * \brief convert DWORD (unsigned int) to vec4
		 * \param d the DWORD (unsigned int)
		 * \return the 4d vector
		 */
		static vec4 dwordToVec4(unsigned int d);

		/**
		 * \brief convert vec4 to DWORD (unsigned int)
		 * \param v the 4d vector
		 * \return the DWORD (unsigned int)
		 */
		static unsigned vec4ToDword(const vec4& v);

		/**
		 * \brief set the random seed for all further random value generation
		 * \param seed the seed
		 */
		static void randomize(int seed);

		/**
		 * \brief get the random value 0..1
		 * \return the random value
		 */
		static float random01();

		/**
		 * \brief get the random value in range
		 * \param min low bound
		 * \param max high bound
		 * \return the random value
		 */
		static float random(float min, float max);

		/**
		 * \brief get the normalized random vector
		 * \return the 3d random vector
		 */
		static vec3 randomNormal();

		/**
		 * \brief returns the perlin noise 3d vector
		 * \param p the value in 3d space
		 * \param seed the seed
		 * \return the perlin noise value
		 */
		static vec3 perlin3d(vec3 p, float seed = 0);

		/**
		 * \brief generate the perlin noise value
		 * \param p the value in 3d space
		 * \param seed the seed
		 * \return the perlin noise value
		 */
		static float perlin(vec3 p, float seed = 0);

		/**
		 * \brief get the value from the global strings list by index. That lists used in dropdown boxes in UI
		 * \param enumID the enumerator ID
		 * \param index the index of the value
		 * \return the string
		 */
		static const char* getEnumValueByIndex(const char* enumID, int index);

		/**
		 * \brief get the integer value that corresponds to the string value from the global strings list.
		 * \param enumID the enumerator ID
		 * \param key the string value fo find
		 * \return the integer value that corresponds to the string
		 */
		static int getEnumValue(const char* enumID, const char* key);

		/**
		 * \brief get the index of the value in the global strings list. That lists used in dropdown boxes in UI
		 * \param enumID the enumerator ID
		 * \param key the value to find
		 * \return the index of the value, -1 means that value not found
		 */
		static int getEnumValueIndex(const char* enumID, const char* key);

		/**
		 * \brief get the count of the values in the global strings list. That lists used in dropdown boxes in UI
		 * \param enumID the enumerator ID
		 * \return the count of the values
		 */
		static int getEnumValuesCount(const char* enumID);

		/**
		 * \brief clear the global strings list.
		 * \param enumID the enumerator ID
		 */
		static void clearEnum(const char* enumID);

		/**
		 * \brief add the value to the global strings list.
		 * \param enumID the enumerator ID
		 * \param key the string to add
		 * \param value the integer value that corresponds to the string, -1 means that the value will be the index of the string in the list, it is default value
		 */
		static void addEnumValue(const char* enumID, const char* key, int value = - 1);

		/**
		 * \brief exit the 3DCoat
		 */
		static void quit();

		/**
		 * \brief report that the test was successful. In this case the file "InstallFolder/.installer/test_success.txt created
		 */
		static void testSuccessful();

		/**
		 * \brief report that the test was successful. In this case the file "InstallFolder/.installer/test_failed.txt created
		 * \param message the message to put into that file
		 */
		static void testFailed(const char* message);

		/**
		 * \brief send some message to 3DCoat (usually used for internal purposes)
		 * \param message the message
		 */
		static void signal(const char* message);
		
		/**
		 * \brief get the list of last signals sent to 3DCoat 
		 * \return the list reference
		 */
		static std::vector<std::string>& last_signals();

		/**
		 * \brief get the current FPS
		 * \return the fps value (averaged)
		 */
		static float getFPS();

		/**
		 * \brief get the frame time in milliseconds
		 * \return the milliseconds amount
		 */
		static float getFrameTimeMs();

		/**
		 * \brief check if the viewport is in render process in render room
		 * \return true if in render
		 */
		static bool inRenderProcess();

		/**
		 * \brief Globally set the value for the key, it is even stored between sessions of the 3DCoat
		 * \param key the key
		 * \param value the value to store
		 */
		static void set(const char* key, const char* value);

		/**
		 * \brief Get previously stored value by the key
		 * \param key the key
		 * \return the value as string, empty string if not found
		 */
		static const char* get(const char* key);
	};

	/**
	 * \brief The UV API. The mesh is taken from the current room. If paint or UV rooms is active, the mesh is taken from the paint room, otherwise the mesh is taken from the Retopo room
	 */
	class APICALL uv {
	public:
		/**
		 * \brief get the UV-sets count.
		 * \return the amount
		 */
		static int uvSetsCount();

		/**
		 * \brief set the border around the islands when we pack it
		 * \param distance the border size in percents
		 */
		static void setUnwrapIslandsDistance(float distance);

		/**
		 * \brief get the border around the islands when we pack it
		 * \return the border size in percents
		 */
		static float getUnwrapIslandsDistance();

		/**
		 * \brief get the current uv-set index
		 * \return the index
		 */
		static int currentUvSet();

		/**
		 * \brief get the islands count over the current uv-set
		 * \param uv_set the uv-set index
		 * \return teh islands count
		 */
		static int islandsCount(int uv_set);

		/**
		 * \brief get the mesh that contains the island, xy of each point is the UV coordinate. The mesh contains only one island
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \return the flat mesh
		 */
		static Mesh islandToMesh(int uv_set, int island_index);

		/**
		 * \brief get the mesh that contains the island, each point is the coordinate in space (not the uv coordinate!). The mesh contains only one island. The faces correspond to the faces of the mesh that was got by islandToMesh
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \returns mesh the 3D mesh
		 */
		static Mesh islandToMeshInSpace(int uv_set, int island_index);

		/**
		 * \brief get the mapping from the vertex index in the mesh that was got by islandToMesh to the vertex index in the original mesh
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \return the list of the positional vertex indices of the original mesh in same order as the vertices in the mesh that was got by islandToMesh
		 */
		static std::vector<int> getIslandVertexMapping(int uv_set, int island_index);

		/**
		 * \brief get unsorted list of edges on the border of the island
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \return the list of edges, even amount of elements, each pair of elements is the positional vertex indices of the original mesh
		 */
		static std::vector<int> getIslandBorder(int uv_set, int island_index);

		/**
		 * \brief get the border between two islands
		 * \param uv_set1 the uv set index of the first island
		 * \param island_index1 the island index within the uv set of the first island
		 * \param uv_set2 the uv set index of the second island
		 * \param island_index2 the island index within the uv set of the second island
		 * \return the list of edges that are common for both islands, even amount of elements, each pair of elements is the positional vertex indices of the original mesh
		 */
		static std::vector<int> getBorderBetweenIslands(int uv_set1, int island_index1, int uv_set2, int island_index2);

		/**
		 * \brief get the uv coordinate of the positional vertex in the island
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \param vertex_index the positional vertex index
		 * \return the uv coordinate of the vertex, vec2(0,0) if the vertex is not in the island
		 */
		static vec2 getIslandVertexUv(int uv_set, int island_index, int vertex_index);

		/**
		 * \brief Flatten the mesh that consists of the single island
		 * \param mesh the mesh that consists of the single island
		 * \param method the flattening method. 0 - flatten to the plane, 1 - LSCM, 2 - ABF, 3 - GU, 4 - Stripe (if possible)
		 * \param optimize_rotation optimize the rotation of the island, place it approximately horizontally or vertically
		 * \param scale_to_geometry scale the island to keep average edge length equal to the average edge length of the original mesh
		 * \return the flat mesh
		 */
		static Mesh flattenSingleIsland(const Mesh& mesh, int method, bool optimize_rotation = true, bool scale_to_geometry = true);

		/**
		 * \brief use the mesh (that was previously got by islandToMesh) to replace the island in the current uv-set
		 * \param mesh the mesh that was previously got by islandToMesh
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		static void meshToIsland(const Mesh& mesh, int uv_set, int island_index);

		/**
		 * \brief pack the islands in the current uv-set
		 * \param uv_set the uv set index
		 * \param rotate allow rotation while packing
		 * \param shuffle shuffle the identical islands to avoid the exact overlapping
		 */
		static void pack(int uv_set, bool rotate, bool shuffle);

		/**
		 * \brief unwrap the current uv-set
		 * \param uv_set the uv set index
		 */
		static void unwrap(int uv_set);

		/**
		 * \brief unwrap the island using the ABF approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		static void toAbf(int uv_set, int island_index);

		/**
		 * \brief unwrap the island using the LSCM approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		static void toLscm(int uv_set, int island_index);

		/**
		 * \brief unwrap the island using the GU (Globally Uniform) approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		static void toGu(int uv_set, int island_index);

		/**
		 * \brief unwrap the island using the Planar approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		static void toPlanar(int uv_set, int island_index);

		/**
		 * \brief try to uwrap the island as the regular stripe
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		static void toStripe(int uv_set, int island_index);

		/**
		 * \brief move the island from one uv-set to another one
		 * \param uv_set the source uv set index
		 * \param island_index the island index within the source uv set
		 * \param destination_uv_set the destination uv set index
		 */
		static void toUvSet(int uv_set, int island_index, int destination_uv_set);

		/**
		 * \brief get the whole mesh from the paint/UV/Retopo room - in dependence on current room
		 * \return  the whole paint or retopo mesh (in dependence on current room)
		 */
		static Mesh getWholeMesh();

		/**
		 * \brief get the selected faces as the Mesh object
		 * \return the Mesh 
		 */
		static Mesh selectedToMesh();

		/**
		 * \brief get all seams across the mesh
		 * \return the list of integer values, each value is the index of the vertex in the mesh, the even index is start of the seam, the odd index is the end of the seam
		 */
		static std::vector<int> getSeams();

		/**
		 * \brief add the seam to the mesh
		 * \param start_vertex_index the start positional vertex index
		 * \param end_vertex_index the end positional vertex index
		 */
		static void addSeam(DWORDS2 start_vertex_index, int end_vertex_index);

		/**
		 * \brief remove the seam from the mesh
		 * \param start_vertex_index the start positional vertex index
		 * \param end_vertex_index the end positional vertex index
		 */
		static void removeSeam(int start_vertex_index, int end_vertex_index);

		/**
		 * \brief get the sharp edges across the mesh
		 * \return the list of integer values, each value is the index of the vertex in the mesh, the even index is start of the edge, the odd index is the end of the edge
		 */
		static std::vector<int> getSharpEdges();

		/**
		 * \brief add the sharp edge to the mesh
		 * \param start_vertex_index the start positional vertex index
		 * \param end_vertex_index the end positional vertex index
		 */
		static void addSharpEdge(int start_vertex_index, int end_vertex_index);

		/**
		 * \brief remove the sharp edge from the mesh
		 * \param start_vertex_index the start positional vertex index
		 * \param end_vertex_index the end positional vertex index
		 */
		static void removeSharpEdge(int start_vertex_index, int end_vertex_index);

		/**
		 * \brief re-wrap/extend  islands in correspondence to the changed seams and inserted faces. Pay attention, that it may lead to islands intersection.
		 */
		static void unwrapUnassigned();

		/**
		 * \brief apply uv changes to the paint room mesh (if we use uv/paint context)
		 */
		static void applyUVSet();
	};


	/**
	 * \brief the parameters to be used for cluster scaling in Model, scaleSelectedFacesClusters
	 */
	enum ClusterScale
	{
		Uniform_Scaling = 0,
		Axial_Normal = 1,
		Axial_X = 2,
		Axial_Y = 3,
		Axial_Z = 4,
		Radial_Normal = 5,
		Radial_X = 6,
		Radial_Y = 7,
		Radial_Z = 8
	};
	/**
	 * \brief The class that corresponds to the retopo/modeling rooms meshes. This is advanced version of the Mesh that allows essential topology changes on the fly.
	 * It is very similar to Mesh by basic functionality, may be easily converted to Mesh and vice versa. But it is more heavy structure, use Mesh if you don't need the advanced functionality.
	 * 
	 */
	class APICALL Model {
	protected:
		friend class Mesh;
		ClusteredMesh* _mesh;
		bool _from_retopo;
		bool _from_uv;
		ClusteredMesh& _mctx();
		ClusteredMesh* _pmctx() const;
		static void inc_ref(ClusteredMesh* m);
		static void dec_ref(ClusteredMesh* m);
		static bool has_ref(ClusteredMesh* m);
		static ClusteredMesh* allocate();
		static ClusteredMesh* allocate(ClusteredMesh* src);
		static std::vector<std::pair<ClusteredMesh*, int>> allocated_meshes;
	public:
		/**
		 * \brief construct the lowpoly mesh, it is completely independent on retopo/modeling/uv workspace. If you need to refer any of that workspace use static functions fromRetopo, fromModeling, fromUV
		 */
		Model();

		/**
		 * \brief make a reference to the source mesh. No new Mesh allocated! Use MakeCopy if you need to make a copy of the mesh
		 * \param source the source mesh
		 */
		Model(const Model& source);

		/**
		 * \brief construct model from the Mesh object.
		 * \param source the source Mesh object
		 */
		Model(const Mesh& source);

		/**
		 * \brief the assignment operator. No new Mesh allocated! Use MakeCopy if you need to make a copy of the mesh
		 * \param source the source mesh
		 * \return the Mesh reference
		 */
		Model & operator=(const Model& source);

		Model & operator=(const Mesh& source);

		Model & operator += (const Model& source);

		Model & operator += (const Mesh& source);

		/**
		 * \brief transform the whole Model with the matrix
		 * \param m the transformation matrix
		 * \return the reference to the Model
		 */
		Model& transform(const mat4& m);

		/**
		 * \brief make a copy of the source mesh. Pay attention, if you taken it from the retopo/uv context, it will no longer refer to the retopo/uv mesh, it will be independent copy
		 */
		Model MakeCopy() const;

		/**
		 * \brief destroy the mesh reference (the mesh itself will not be destroyed if it is the reference to retopo/modeling/uv workspaces)
		 */
		~Model();

		/**
		 * \brief get the reference to the mesh in the retopo room
		 * \return the reference to the mesh
		 */
		static Model fromRetopo();

		/**
		 * \brief get the reference to the mesh in the modeling room, currently it is the same mesh as in the retopo room
		 * \return the reference to the mesh
		 */
		static Model fromModeling();

		/**
		 * \brief get the reference to the mesh in the uv room, pay attention that topology changes to that mesh may lead to instability!
		 * \return the reference to the mesh
		 */
		static Model fromUv();

		/**
		 * \brief Set the display options for the retopo/modeling/uv meshes
		 * \param showWireframe show the wireframe
		 * \param showColored show colored clusters
		 * \param showSeams show seams
		 * \param showSharpEdges show sharp edges
		 * \param smoothView smooth view
		 */
		void displayOptions(bool showWireframe = true, bool showColored = true, bool showSeams = true, bool showSharpEdges = true, bool smoothView = false);

		/**
		 * \brief get the retopo groups count
		 * \return the amount
		 */
		int getObjectsCount();

		/**
		 * \brief get the index of the current group
		 * \return the index
		 */
		int getCurrentObject();

		/**
		 * \brief set the current group index
		 * \param index the index
		 */
		void setCurrentObject(int index);

		/**
		 * \brief get the retopo group name
		 * \param group_index the group index
		 * \return the name
		 */
		const char* getObjectName(int group_index);

		/**
		 * \brief remove the group by index
		 */
		void removeObject(int group_index);

		/**
		 * \brief rename the group by index
		 * \param index the group index to rename
		 * \param name the new name
		 */
		void setObjectName(int index, const char* name);

		/**
		 * \brief set the group visibility
		 * \param index the group index
		 * \param visible the visibility state
		 */
		void setObjectVisibility(int index, bool visible);

		/**
		 * \brief get the group visibility
		 * \param index the group index
		 * \return the visibility state
		 */
		bool getObjectVisibility(int index);

		/**
		 * \brief add new retopo group
		 * \param name the group name
		 * \return the index of new group
		 */
		int addObject(const char* name);

		/**
		 * \brief add the new UV set/Material
		 * \param name the name
		 * \return the new UV set/Material index
		 */
		int addMaterial(const char* name);

		/**
		 * \brief remove all unused UV sets (not referred within the mesh)
		 */
		void removeUnusedMaterials();

		/**
		 * \brief get the group reference color
		 * \param group_index the group index
		 * \return the (r,g,b,a) vector, 0..255
		 */
		vec4 getObjectReferenceColor(int group_index);

		/**
		 * \brief set the group reference color
		 * \param group_index the group index
		 * \param color the (r,g,b,a) vector, 0..255
		 */
		void setObjectReferenceColor(int group_index, vec4 color);

		/**
		 * \brief move the selected faces to the group
		 * \param group_index the group index
		 */
		void selectedToObject(int group_index);

		/**
		 * \brief get the whole mesh from the retopo room
		 * \return the Mesh object
		 */
		Mesh getWholeMesh();

		/**
		 * \brief get the selected faces as the Mesh object
		 * \return the Mesh
		 */
		Mesh selectedToMesh();

		/**
		 * \brief get the visible faces as the Mesh object
		 * \return teh Mesh
		 */
		Mesh visibleToMesh();

		/**
		 * \brief insert the mesh to the retopo/modeling room, each object of the mesh treated as the new retopo layer 
		 * \param mesh the Mesh object
		 * \param Transform the transformation matrix
		 * \param b the boolean operation type
		 * \param select the flag that indicates if we need to select faces of the the inserted mesh, used only if b is BOOL_MERGE
		 * \param snap_to_existing the flag that indicates if we need to snap the mesh to the existing sculpt/paint objects
		 */
		void addTransformed(const Mesh& mesh, const mat4& Transform = mat4::Identity, BoolOpType b = BOOL_MERGE, bool select = false, bool snap_to_existing = false);

		/**
		 * \brief get the mesh from some retopo group
		 * \param group_index the group index
		 * \return the Mesh object
		 */
		Mesh getObjectMesh(int group_index);

		/**
		 * \brief replace the retopo layer with mesh
		 * \param group_index the group index
		 * \param mesh the Mesh object to insert
		 * \param transform the transformation matrix
		 */
		void setObjectMesh(int group_index, Mesh& mesh, const mat4& transform = mat4::Identity);

		/**
		 * \brief duplicate the object (retopo group)
		 * \param group_index the object/group index
		 * \param name the new name, if not passed the name will be generated automatically
		 * \param transform the additional transformation matrix
		 * \param select the flag that indicates if we need to select the new object's faces (in addition to existing selection) 
		 * \return the new object index
		 */
		int duplicateObject(int group_index, const char* name = nullptr, const mat4& transform = mat4::Identity, bool select = false);

		/**
		 * \brief generate unique name for the object, it will start as the string in base base
		 * \param base the base name
		 * \return the unique name
		 */
		std::string generateName(const char* base);

		/**
		 * \brief remove all faces from the group
		 * \param group_index the group index
		 */
		void clearObjectMesh(int group_index);

		/**
		 * \brief clear the whole mesh
		 */
		void clear();

		/**
		 * \brief Drop the whole mesh to the undo queue, it is important if you want allow the user to undo your mesh changes, call it before your changes. It works for UV room too.
		 */
		void dropUndo();

		/**
		 * \brief get the list of selected faces
		 * \return the list of selected faces
		 */
		std::vector<int> getSelectedFaces();
		void getSelectedFaces(list<int>& faces);

		/**
		 * \brief set the selected faces list
		 * \param faces the faces indices list
		 */
		void setSelectedFaces(std::vector<int>& faces);
		void setSelectedFaces(list<int>& faces);

		/**
		 * \brief select the face by index
		 * \param face the face index
		 */
		void selectFace(int face);

		/**
		 * \brief select all feces in the group
		 * \param group_index the group index
		 */
		void selectObject(int group_index, bool add_to_selected = true);

		/**
		 * \brief get the list of faces in the group
		 * \param group_index the group index
		 * \return the list of faces
		 */
		std::vector<int> getObjectFaces(int group_index);
		void getObjectFaces(int group_index, list<int>& faces);

		/**
		 * \brief check if the face selected
		 * \param face the face index
		 * \return the selection state
		 */
		bool isFaceSelected(int face);

		/**
		 * \brief unselect all faces
		 */
		void unselectAllFaces();

		/**
		 * \brief expand the faces/vertices/edges selection to the connected geometry
		 */
		void expandSelection();

		/**
		 * \brief contract the faces/vertices/edges selection to the connected geometry
		 */
		void contractSelection();

		/**
		 * \brief convert faces/vertices selection to edges selection
		 */
		void selectedToEdges();

		/**
		 * \brief convert edges/vertices selection to faces selection
		 */
		void selectedToFaces();

		/**
		 * \brief convert faces/edges selection to vertices selection
		 */
		void selectedToVertices();

		/**
		 * \brief returns even amount of vertex indices, pairs os start and end vertices of the selected edges
		 * \return the list of vertex indices (pairs)
		 */
		std::vector<int> getSelectedEdges();
		void getSelectedEdges(list<int>& edges);

		/**
		 * \brief set the selected edges list
		 * \param edges the edges indices list (should be even amount of indices)
		 */
		void setSelectedEdges(std::vector<int>& edges);
		void setSelectedEdges(list<int>& edges);

		/**
		 * \brief select the edge by vertex indices (add to selection)
		 * \param vertex1 the first vertex index
		 * \param vertex2 the second vertex index
		 */
		void selectEdge(int vertex1, int vertex2);

		/**
		 * \brief check if the edge is selected, order of vertices has no matter
		 * \param vertex1 the first vertex index
		 * \param vertex2 the second vertex index
		 * \return true if the edge is selected
		 */
		bool isEdgeSelected(int vertex1, int vertex2);

		/**
		 * \brief unselect all edges
		 */
		void unselectAllEdges();

		/**
		 * \brief get the list of selected vertices
		 * \return the list of selected vertices
		 */
		std::vector<int> getSelectedVertices();
		void getSelectedVertices(list<int>& vertices);

		/**
		 * \brief get the soft selection weights of the selected vertices, 1 is maximum value
		 * \return the list of weights, the size of the list is equal to the size of the selected vertices list
		 */
		std::vector<float> getSelectedVerticesWeights();

		/**
		 * \brief set the selected vertices list
		 * \param vertices the list of vertices indices
		 * \param weights the list of soft selection weights, the size of the list should be zero or equal to the size of the vertices list. If it is empty, the vertices will be selected with the maximal weight 
		 */
		void setSelectedVertices(const std::vector<int>& vertices, const std::vector<float>& weights);
		void setSelectedVertices(const list<int>& vertices, const list<float>& weights);

		/**
		 * \brief add the vertex to the selection
		 * \param vertex the vertex index
		 * \param weight the soft selection weight, 1 is maximum value
		 */
		void selectVertex(int vertex, float weight = 1.0f);

		/**
		 * \brief check if the vertex is selected
		 * \param vertex the vertex index
		 * \return true if the vertex is selected
		 */
		bool isVertexSelected(int vertex);

		/**
		 * \brief unselect all vertices
		 */
		void unselectAllVertices();

		/**
		 * \brief get the faces count
		 * \return the faces count
		 */
		int facesCount();

		/**
		 * \brief get the positional vertices count
		 * \return the vertices count
		 */
		int vertsCount();

		/**
		 * \brief get the uv vertices count
		 * \return the uv vertices count
		 */
		int vertsUvCount();

		/**
		 * \brief remove the face by index
		 * \param face the face index
		 */
		void removeFace(int face);

		/**
		 * \brief create empty face, you need to call setFaceVertices to set the vertices, setFaceUVVerts to set the UV vertices
		 * \param Group the face group index
		 * \param UVSet the UV set index
		 * \return the new face index
		 */
		int createNewFace(int Group, int UVSet);

		/**
		 * \brief get the vertices count over the face
		 * \param face the face index
		 * \return the vertices count
		 */
		int getFaceVertsCount(int face);

		/**
		 * \brief get the vertex index over the face
		 * \param face the face index
		 * \param vertex_index the vertex index over the face
		 * \return the vertex index, -1 if the vertex/face index is out of range
		 */
		int getFaceVertex(int face, int vertex_index);

		/**
		 * \brief get the list of UV vertex indices over the face, pay attention UV vertices are not same as position vertices
		 * \param face the face index
		 * \return the list of vertex indices
		 */
		std::vector<int> getFaceVerts(int face);
		void getFaceVerts(int face, list<int>& vertices);

		/**
		 * \brief set the list of positional vertex indices over the face
		 * \param face the face index
		 * \param vertices the list of vertex indices
		 */
		void setFaceVerts(int face, const std::vector<int>& vertices);
		void setFaceVerts(int face, const list<int>& vertices);

		/**
		 * \brief get the face visibility
		 * \param face the face index
		 * \return the visibility state
		 */
		bool getFaceVisibility(int face);

		/**
		 * \brief set the face visibility
		 * \param face the face index
		 * \param visibility the visibility state
		 */
		void setFaceVisibility(int face, bool visibility);

		/**
		 * \brief get the face square
		 * \param face the face index
		 * \return the square
		 */
		float getFaceSquare(int face);

		/**
		 * \brief get the face square in UV space
		 * \param face the face index
		 * \return the square
		 */
		float getFaceUVSquare(int face);

		/**
		 * \brief get the face normal
		 * \param face the face index
		 * \return the face normal
		 */
		vec3 getFaceNormal(int face);

		/**
		 * \brief get the group index of the face
		 * \param face the face index
		 * \return the group index
		 */
		int getFaceObject(int face);

		/**
		 * \brief set the group index of the face
		 * \param face the face index
		 * \param group the group index
		 */
		void setFaceObject(int face, int group);

		/**
		 * \brief get the UV set index for the face
		 * \param face the face index
		 * \return the UV set index, -1 if out of range
		 */
		int getFaceMaterial(int face);

		/**
		 * \brief set the UV set for the face
		 * \param face the face index
		 * \param uv_set the UV set index
		 */
		void setFaceMaterial(int face, int uv_set);

		/**
		 * \brief get the amount of UV vertices over the face
		 * \param face the face index
		 * \return amount of vertices over the face, 0 if UV-s not assigned
		 */
		int getFaceUvVertsCount(int face);

		/**
		 * \brief get the UV vertex index over the face
		 * \param face the face index
		 * \param vertex_index the vertex index over the face
		 * \return the UV vertex index, -1 if the vertex/face index is out of range
		 */
		int getFaceUvVertex(int face, int vertex_index);

		/**
		 * \brief get the list of UV vertices indices over the face
		 * \param face the face index
		 * \return the list of UV vertices indices
		 */
		std::vector<int> getFaceUvVerts(int face);
		void getFaceUvVerts(int face, list<int>& vertices);

		/**
		 * \brief set the UV vertices for the face
		 * \param face the face index
		 * \param vertices the UV vertices list
		 */
		void setFaceUvVerts(int face, const std::vector<int>& vertices);
		void setFaceUvVerts(int face, const list<int>& vertices);

		/**
		 * \brief get the vertex position in space
		 * \param vertex the vertex index
		 * \return the position
		 */
		vec3 getVertex(int vertex);

		/**
		 * \brief set the vertex position in space
		 * \param vertex the vertex index
		 * \param position the position
		 */
		void setVertex(int vertex, const vec3& position);

		/**
		 * \brief create the positional vertex
		 * \param position the position
		 * \return the positional vertex index
		 */
		int createNewVertex(const vec3& position);

		/**
		 * \brief get the UV coordinates of the UV vertex
		 * \param uv_vertex the uv vertex index
		 * \return teh UV coordinates
		 */
		vec2 getVertexUV(int uv_vertex);

		/**
		 * \brief set the UV for the UV vertex
		 * \param uv_vertex the uv vertex index
		 * \param uv the UV coordinates
		 */
		void setVertexUV(int uv_vertex, const vec2& uv);

		/**
		 * \brief create new UV vertex to be used for faces
		 * \param uv the texture coordinates
		 * \return the index
		 */
		int createNewUvVertex(const vec2& uv);

		/**
		 * \brief get vertex normal, calculated as average of adjacent faces normals
		 * \param vertex the vertex index
		 * \return the normal
		 */
		vec3 getVertexNormal(int vertex);

		/**
		 * \brief update the vertex normals
		 * \param for_snapping if true, the normals will lay in the middle of faces, ne respecting the faces square.
		 */
		void updateNormals(bool for_snapping = true);

		/**
		 * \brief update the connectivity information, it should be called sometimes if you feel that the connectivity information lost due to some heavy operations
		 */
		void updateTopology();

		/**
		 * \brief complete cleanul from non-manifolds or other problems, some faces may be removed
		 */
		void cleanup();

		/**
		 * \brief get the list of vertices that are adjacent to the vertex
		 * \param vertex the vertex index
		 * \return the list of adjacent vertices
		 */
		std::vector<int> getVertsNearVertex(int vertex);
		void getVertsNearVertex(int vertex, list<int>& vertices);

		/**
		 * \brief get the list of faces that are adjacent to the vertex
		 * \param vertex the vertex index
		 * \return the list of adjacent faces
		 */
		std::vector<int> getFacesNearVertex(int vertex);
		void getFacesNearVertex(int vertex, list<int>& faces);

		/**
		 * \brief get the list of faces that are adjacent to the face
		 * \param face the face index
		 * \return the list of adjacent faces
		 */
		std::vector<int> getFaceNeighbors(int face);
		void getFaceNeighbors(int face, list<int>& faces);

		/**
		 * \brief get the list of faces that are adjacent to the edge
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \return the list of adjacent faces
		 */
		std::vector<int> getFacesNearEdge(int vertex1, int vertex2);
		void getFacesNearEdge(int vertex1, int vertex2, list<int>& faces);

		/**
		 * \brief check if the edge is open
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \return true if open
		 */
		bool isOpenEdge(int vertex1, int vertex2);

		/**
		 * \brief check if the edge is sharp
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \return true if sharp
		 */
		bool isSharpEdge(int vertex1, int vertex2);

		/**
		 * \brief set the sharpness state for the edge 
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \param sharp the sharpness state
		 */
		void setEdgeSharpness(int vertex1, int vertex2, bool sharp);

		/**
		 * \brief check if edge is seam
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \return true if seam
		 */
		bool isSeam(int vertex1, int vertex2);

		/**
		 * \brief set or clear the seam state for the edge
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \param seam the seam state
		 */
		void setEdgeSeam(int vertex1, int vertex2, bool seam);

		/**
		 * \brief collapse the edge to the middle of the edge
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 */
		void collapseEdge(int vertex1, int vertex2);

		/**
		 * \brief get the islands count over the current uv-set
		 * \param uv_set the uv-set index
		 * \return teh islands count
		 */
		int islandsCount(int uv_set);

		/**
		 * \brief get the mesh that contains the island, xy of each point is the UV coordinate. The mesh contains only one island
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \return the flat mesh
		 */
		Mesh islandToMesh(int uv_set, int island_index);

		/**
		 * \brief get the mesh that contains the island, each point is the coordinate in space (not the uv coordinate!). The mesh contains only one island. The faces correspond to the faces of the mesh that was got by islandToMesh
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \returns mesh the 3D mesh
		 */
		Mesh islandToMeshInSpace(int uv_set, int island_index);

		/**
		 * \brief get the mapping from the vertex index in the mesh that was got by islandToMesh to the vertex index in the original mesh
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \return the list of the positional vertex indices of the original mesh in same order as the vertices in the mesh that was got by islandToMesh
		 */
		std::vector<int> getIslandVertexMapping(int uv_set, int island_index);

		/**
		 * \brief get unsorted list of edges on the border of the island
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \return the list of edges, even amount of elements, each pair of elements is the positional vertex indices of the original mesh
		 */
		std::vector<int> getIslandBorder(int uv_set, int island_index);

		/**
		 * \brief get the border between two islands
		 * \param uv_set1 the uv set index of the first island
		 * \param island_index1 the island index within the uv set of the first island
		 * \param uv_set2 the uv set index of the second island
		 * \param island_index2 the island index within the uv set of the second island
		 * \return the list of edges that are common for both islands, even amount of elements, each pair of elements is the positional vertex indices of the original mesh
		 */
		std::vector<int> getBorderBetweenIslands(int uv_set1, int island_index1, int uv_set2, int island_index2);

		/**
		 * \brief get the uv coordinate of the positional vertex in the island
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 * \param vertex_index the positional vertex index
		 * \return the uv coordinate of the vertex, vec2(0,0) if the vertex is not in the island
		 */
		vec2 getIslandVertexUv(int uv_set, int island_index, int vertex_index);

		/**
		 * \brief Flatten the mesh that consists of the single island
		 * \param mesh the mesh that consists of the single island
		 * \param method the flattening method. 0 - flatten to the plane, 1 - LSCM, 2 - ABF, 3 - GU, 4 - Stripe (if possible)
		 * \param optimize_rotation optimize the rotation of the island, place it approximately horizontally or vertically
		 * \param scale_to_geometry scale the island to keep average edge length equal to the average edge length of the original mesh
		 * \return the flat mesh
		 */
		static Mesh flattenSingleIsland(const Mesh& mesh, int method, bool optimize_rotation = true, bool scale_to_geometry = true);

		/**
		 * \brief use the mesh (that was previously got by islandToMesh) to replace the island in the current uv-set
		 * \param mesh the mesh that was previously got by islandToMesh
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		void meshToIsland(const Mesh& mesh, int uv_set, int island_index);

		/**
		 * \brief pack the islands in the current uv-set
		 * \param uv_set the uv set index
		 * \param rotate allow rotation while packing
		 * \param shuffle shuffle the identical islands to avoid the exact overlapping
		 */
		void pack(int uv_set, bool rotate, bool shuffle);

		/**
		 * \brief unwrap the current uv-set
		 * \param uv_set the uv set index
		 */
		void unwrap(int uv_set);

		/**
		 * \brief unwrap the island using the ABF approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		void toAbf(int uv_set, int island_index);

		/**
		 * \brief unwrap the island using the LSCM approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		void toLscm(int uv_set, int island_index);

		/**
		 * \brief unwrap the island using the GU (Globally Uniform) approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		void toGu(int uv_set, int island_index);

		/**
		 * \brief unwrap the island using the Planar approach
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		void toPlanar(int uv_set, int island_index);

		/**
		 * \brief try to uwrap the island as the regular stripe
		 * \param uv_set the uv set index
		 * \param island_index the island index within the uv set
		 */
		void toStripe(int uv_set, int island_index);

		/**
		 * \brief Extrude the selected edges or selected faces without the actual moving of the extruded elements. They stay selected, so you amy apply some transform to the selected elements
		 */
		void extrudeSelected();

		/**
		 * \brief move selected faces along the faces normals, trying to keep faces parallel to the original direction 
		 * \param displacement the displacement value
		 */
		void moveSelectedFacesAlongFacesNormals(float displacement);

		/**
		 * \brief move selected faces along the vertex normals, each vertex displace on the same distance
		 * \param displacement the displacement value
		 */
		void moveSelectedFacesAlongVertexNormals(float displacement);

		/**
		 * \brief subdivide the selected faces
		 * \param apply_catmull_clark apply the catmull-clark subdivision
		 */
		void subdivideSelectedFaces(bool apply_catmull_clark = false);

		/**
		 * \brief subdivide the whole mesh
		 * \param apply_catmull_clark apply the catmull-clark subdivision
		 */
		void subdivide(bool apply_catmull_clark = true);

		/**
		 * \brief apply the transformation to the selected elements
		 * \param transform the transformation matrix
		 * \param apply_symmetry apply the global symmetry
		 */
		void transformSelected(const mat4& transform, bool apply_symmetry);

		/**
		 * \brief scale each selection cluster separately, to own center mass
		 * \param scale the scale coefficient
		 */
		void scaleSelectedFacesClusters(float scale, ClusterScale method = Uniform_Scaling);

		/**
		 * \brief perform the bevel over the selected vertices. As result, new faces will be selected
		 * \param size the bevel size
		 */
		void bevelOverSelectedVertices(float size);

		/**
		 * \brief perform the bevel over the selected edges.
		 * \param size the bevel width
		 * \param OldVariant if true the older variant of the bevel (splits edges in strightforward way), in some cases it works more stable.
		 */
		void bevelOverSelectedEdges(float size, int segments = 1, bool OldVariant = false);

		/**
		 * \brief split existing edge somewhere between vertices.
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \param position the position to split the edge, [0..1], 0 - near the vertex1, 1 - near the vertex2
		 * \return the new vertex index
		 */
		int splitEdge(int vertex1, int vertex2, float position);

		/**
		 * \brief split existing edge somewhere between vertices.
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \return true if succeeed to connect
		 */
		bool connect(int vertex1, int vertex2);

		/**
		 * \brief check if connecting the two vertices is possible
		 * \param vertex1 the positional vertex index (1)
		 * \param vertex2 the positional vertex index (2)
		 * \return true if connection is possible
		 */
		bool checkConnectivity(int vertex1, int vertex2);

		/**
		 * \brief connect selected vertices in smart way
		 */
		void connectSelectedVerts();

		/**
		 * \brief invert selected faces only within the connective area, if some objects has no selected faces, the selection there will not change
		 */
		void invertSelectedFacesTopoplogically();

		/**
		 * \brief perform the inset over the selected faces
		 */
		void inset(float distance);

		/**
		 * \brief perform the shell operation over the selected faces. After calling the shell() you should call the moveSelectedFacesAlongFacesNormals or moveSelectedFacesAlongVertexNormals
		 * to give some thickness to the resulting figure 
		 */
		void shell();

		/**
		 * \brief perform the intrude operation over the selected faces. After calling the intrude() you should call the moveSelectedFacesAlongFacesNormals or moveSelectedFacesAlongVertexNormals
		 * to give some thickness to the resulting figure
		 */
		void intrude();

		/**
		 * \brief relax selected vergtices
		 */
		void relaxSelected();

		/**
		 * \brief select all edges on the path from vertex1 to vertex2 (add to existing edges selection)
		 * \param vertex1 the first vertex
		 * \param vertex2 the second vertex
		 */
		void selectPath(int vertex1, int vertex2);

		/**
		 * \brief get all vertices on the path from vertex1 to vertex2
		 * \param vertex1 the first vertex
		 * \param vertex2 the second vertex
		 */
		std::vector<int> getPath(int vertex1, int vertex2);
		
	};
	class APICALL logger {
		FILE* out;
		void* tempTimer;
		str filename;
		str _accum;
		int precission;
		void append(const char*);
		void close();
		
	public:
		/**
		 * \brief create the logger object. You may call directTo later and flush the output there.
		 * Otherwise, the output will be directed to the Log.txt in case if you call flush();
		 */
		logger();

		/**
		 * \brief create the logger object and direct the output there
		 * \param filename the filename to direct
		 */
		logger(const char* filename);


		~logger();
		/**
		 * \brief open the accumulated log in the default text editor
		 */
		void open();

		/**
		 * \brief show the accumulated log in the message box
		 */
		void showMessage();

		/**
		 * \brief Direct the log output to the file  
		 * \param filename the filename to log there. The filename may be relative path.
		 * \return the logger reference
		 */
		logger& directTo(const char* filename);

		/**
		 * \brief Returns the absolute path to the log file.
		 * \return the absolute path even if you passed the relative path. 
		 */
		str getFullPath();

		/**
		 * \brief send the string to the output
		 * \return the reference
		 */
		logger& operator << (const char*);

		/**
		 * \brief send the string to the output
		 * \return the reference
		 */
		logger& operator << (str&);

		/**
		 * \brief send the Unicode string to the output
		 * \return the reference
		 */
		logger& operator << (const wchar_t*);

		/**
		 * \brief send the integer value to the output
		 * \return the refernce
		 */
		logger& operator << (int);

		/**
		 * \brief send the float value to the output in correspondence with the current precission
		 * \return the reference
		 */
		logger& operator << (float);

		/**
		 * \brief send the double value to the output in correspondence with the current precission
		 * \return the reference
		 */
		logger& operator << (double);

		/**
		 * \brief send the 2d vector to the output
		 * \return the reference
		 */
		logger& operator << (const vec2&);

		/**
		 * \brief send the 3d vector to the output
		 * \return the reference
		 */
		logger& operator << (const vec3&);

		/**
		 * \brief send the 4d vector to the output
		 * \return the reference
		 */
		logger& operator << (const vec4&);

		/**
		 * \brief send the BaseClass to the output as json
		 * \return the reference
		 */
		logger& operator << (BaseClass*);

		/**
		 * \brief send the formatted message 
		 * \param format 
		 * \param ... 
		 * \return the reference
		 */
		logger& format(const char* format, ...);

		/**
		 * \brief save all acumulated text to the file
		 * \return the chain reference
		 */
		logger& flush();

		/**
		 * \brief start newline in the text file
		 * \return the chain reference
		 */
		logger& newline();

		/**
		 * \brief add the timestamp to the log as the 
		 * \return the reference
		 */
		logger& timestamp();

		/**
		 * \brief start the timer to profile some operation
		 * \return the reference
		 */
		logger& startTimer();

		/**
		 * \brief stop the timer and output the time into the log as amount of microseconds
		 * \return the reference
		 */
		logger& endTimer();

		/**
		 * \brief set the precission of floating-point output
		 * \return the reference
		 */
		logger& floatPrecission(int signs = 2);
	};


}


#include "CorePrimAPI.h"
#pragma pack(pop)

#ifdef COMMS_OPENGL
#include <ui.h>
#include <ui_definition.h>
#define EXPORT(x) \
	static int Main();\
	__defineui(xxx) {\
		ui_element(#x, Main);\
	} static
#define EXPORT_COMMAND(x) \
	static void x();\
	__defineui(x) {\
		ui_element(#x, x);\
	} static void x()
#define EXPORT_EXTENSION(x) __defineui(___##x)
void executeCoreScript(const char* filename, bool build, bool clean);
#endif //COMMS_OPENGL
