#pragma once

typedef comms::cVec2 cVec2;
class Rct
{
public:
    Rct()                                            : x(0), y(0), w(0), h(0)            {}
    Rct( float _w, float _h )                        : x(0.0f), y(0.0f), w(_w), h(_h)    {}
    Rct( float _x, float _y, float _side )            : x(_x), y(_y), w(_side), h(_side)    {}

    inline Rct( float _x, float _y, float _w, float _h );

    float           b() const { return y + h; }
    float           r() const { return x + w; }
    inline float        GetAspect    ()                            const;
    inline float        MaxSide        ()                            const;
    inline bool        PtIn        ( float pX, float pY )        const;
    inline bool        PtInStrict    ( float pX, float pY )        const;

    inline float        GetCenterX    ()                            const;
    inline float        GetCenterY    ()                            const;
    inline float        Dist2ToPt    ( float pX, float pY )        const;

    inline void        Copy        ( const Rct& orig );
    inline void        Deflate        ( float amt );
    inline void        Inflate        ( float top, float right, float bottom, float left );
    inline void        Inflate        ( float val );

    inline void        FitInto        ( const Rct& rct );
    inline void        CenterInto    ( const Rct& rct );
    
    inline void        Set            ( float _x, float _y, float _w, float _h );
    inline void        Zero        ();
    inline bool        Overlap        ( const Rct& rct ) const;
    inline bool        IsOutside    ( const Rct& rct ) const;
    inline void        Union        ( const Rct& rct );
	inline void		 AddPoint	  (float x,float y);
	inline void		 SetEmpty();
	inline bool		 isEmpty();

    inline float        GetRight    () const;
    inline float        GetBottom    () const;
    inline bool        ClipSegment    ( comms::cVec2& a, comms::cVec2& b ) const;
    inline void        SetPositiveDimensions();

    inline bool         ClipHLine    ( float& px1, float& px2, float py ) const;
    inline bool         ClipVLine    ( float px, float& py1, float& py2 ) const;
    inline bool         Clip        ( Rct& rct ) const;
    inline bool        IsRectInside( float px, float py, float pw, float ph ) const;


    inline void        operator *=( float val );
    inline void        operator /=( float val );
    inline void        operator +=( const comms::cVec2& delta );
    inline void        operator +=( const Rct& delta );


    bool operator == ( const Rct& v ) const
    {
        return (x == v.x)&&(y == v.y)&&(w == v.w)&&(h == v.h);
    }
    
    float                x, y, w, h;

    static const Rct    unit;
    static const Rct    null;
};  // class Rct

/*****************************************************************
/*  File:   Math2D.inl                                           *
/*  Author: Silver, Copyright (C) GSC Game World                 *
/*  Date:   January 2002                                         *
/*****************************************************************/
#define c_SmallEpsilon 1e-20
/*****************************************************************
/*    Rct implementation
/*****************************************************************/
inline Rct::Rct( float _x, float _y, float _w, float _h )
{
    x = _x;
    y = _y;
    w = _w;
    h = _h;
    //if (w < 0.0f) { x += w; w = -w; }
    //if (h < 0.0f) { y += h; h = -h; }
} // Rct::Rct

inline float Rct::GetAspect() const
{
    if (fabs( w ) < c_SmallEpsilon && fabs( h ) < c_SmallEpsilon) return 1.0f;
     return w / h;
}

inline float Rct::GetRight() const
{
    return x + w;
}

inline float Rct::GetBottom() const
{
    return y + h;
}

inline float Rct::GetCenterX() const
{
    return x + w*0.5f;
}

inline float Rct::GetCenterY() const
{
    return y + h*0.5f;
}

inline float Rct::MaxSide()    const
{
    return w > h ? w : h;
}

inline bool Rct::PtIn( float pX, float pY ) const
{
    return  (pX >= x) && (pX <= GetRight()) &&
            (pY >= y) && (pY <= GetBottom());
}

inline bool Rct::PtInStrict( float pX, float pY ) const
{
    return  (pX > x) && (pX < GetRight()) && (pY > y) && (pY < GetBottom());
}

inline void Rct::Copy( const Rct& orig )
{
    x = orig.x;
    y = orig.y;
    h = orig.h;
    w = orig.w;
}

inline void Rct::operator *=( float val )
{
    x *= val;
    y *= val;
    w *= val;
    h *= val;
}

inline void Rct::operator /=( float val )
{
    float iv = 1.0f / val;
    x *= iv;
    y *= iv;
    w *= iv;
    h *= iv;
}

inline void Rct::operator +=( const comms::cVec2& delta )
{
    x += delta.x;
    y += delta.y;
}

inline void Rct::operator +=( const Rct& delta )
{
    x += delta.x;
    y += delta.y;
    w += delta.w;
    h += delta.h;
}

inline void Rct::Set( float _x, float _y, float _w, float _h )
{
    x = _x;
    y = _y;
    w = _w;
    h = _h;
}

inline bool Rct::IsOutside( const Rct& rct ) const
{
    return    (GetRight() < rct.x)    ||
            (x > rct.GetRight())    ||
            (GetBottom() < rct.y)    ||
            (y > rct.GetBottom());
} // Rct::IsOutside

inline bool Rct::Overlap( const Rct& rct ) const
{
    return !IsOutside( rct ) && !rct.IsOutside( *this );

} // Rct::IsOutside

//  fits this rectangle into the rct, constraining the proportion
inline void  Rct::FitInto( const Rct& rct )
{
     float asp = GetAspect();
     if (rct.GetAspect() > asp)
     {
         h = rct.h;
         w = h * asp;
         x = rct.x + (rct.w - w) / 2.0f;
         y = rct.y;
     }
     else
     {
         w = rct.w;
         h = w / asp;
         x = rct.x;
         y = rct.y + (rct.h - h) / 2.0f;
     }
}


inline void Rct::CenterInto( const Rct& rct )
{
    x = rct.x + (rct.w - w)*0.5f;
    y = rct.y + (rct.h - h)*0.5f;
}

inline void Rct::Deflate( float amt )
{
    x -= amt;
    y -= amt;
    w += amt * 2.0f;
    h += amt * 2.0f;
} // Rct::Deflate

inline void Rct::Inflate( float top, float right, float bottom, float left )
{
    x += left;
    y += top;
    w -= left + right;
    h -= top + bottom;
} // Rct::Inflate

inline void Rct::Inflate( float val )
{
    x -= val;
    y -= val;
    w += val * 2.0f;
    h += val * 2.0f;
} // Rct::Inflate

//-------------------------------------------------------------------------------
//  Func:  Rct::ClipSegment
//  Desc:  Clips line segment against rectangle
//  Parm:  a, b - segment endpoints, clipped values are returned there too
//  Ret:   false when segment is totally outside, true when it was clipped
//  Rmrk:  Cohen-Sutherland algorithm was used
//-------------------------------------------------------------------------------
inline bool Rct::ClipSegment(comms::cVec2& a, comms::cVec2& b ) const
{
    float xmax = GetRight();
    float ymax = GetBottom();

    //  fill endpoints status flag bytes
    BYTE stA = 0, stB = 0;
    if (a.y > ymax) stA |= 0x1;
    if (a.y < y)    stA |= 0x2;
    if (a.x > xmax) stA |= 0x4;
    if (a.x < x)    stA |= 0x8;

    if (b.y > ymax) stB |= 0x1;
    if (b.y < y)    stB |= 0x2;
    if (b.x > xmax) stB |= 0x4;
    if (b.x < x)    stB |= 0x8;

    if ((stA & stB) != 0)
    {
        return false;
    }

    if (stA == 0)
    {
        if (stB == 0) return true;
        //    point b is outside rectangle
        if (stB & 0x1)
        //  intersection with ymax
        {
            b.x = a.x + (ymax - a.y)*(b.x - a.x)/(b.y - a.y);
            b.y = ymax;
            return ClipSegment( a, b );
        }

        if (stB & 0x2)
            //  intersection with ymin
        {
            b.x = a.x + (y - a.y)*(b.x - a.x)/(b.y - a.y);
            b.y = y;
            return ClipSegment( a, b );
        }

        if (stB & 0x4)
            //  intersection with xmax
        {
            b.y = a.y + (xmax - a.x)*(b.y - a.y)/(b.x - a.x);
            b.x = xmax;
            return ClipSegment( a, b );
        }

        if (stB & 0x8)
            //  intersection with xmin
        {
            b.y = a.y + (x - a.x)*(b.y - a.y)/(b.x - a.x);
            b.x = x;
            return ClipSegment( a, b );
        }
    }
    else
    {
        if (stA & 0x1)
            //  intersection with ymax
        {
            a.x = b.x + (ymax - b.y)*(a.x - b.x)/(a.y - b.y);
            a.y = ymax;
            return ClipSegment( a, b );
        }

        if (stA & 0x2)
            //  intersection with ymin
        {
            a.x = b.x + (y - b.y)*(a.x - b.x)/(a.y - b.y);
            a.y = y;
            return ClipSegment( a, b );
        }

        if (stA & 0x4)
            //  intersection with xmax
        {
            a.y = b.y + (xmax - b.x)*(a.y - b.y)/(a.x - b.x);
            a.x = xmax;
            return ClipSegment( a, b );
        }

        if (stA & 0x8)
            //  intersection with xmin
        {
            a.y = b.y + (x - b.x)*(a.y - b.y)/(a.x - b.x);
            a.x = x;
            return ClipSegment( a, b );
        }
    }
    
    assert( false );
    return true;
} // Rct::ClipSegment

inline float Rct::Dist2ToPt( float pX, float pY )    const
{
    float x1 = GetRight();
    float y1 = GetBottom();
    float dx = pX < x ? (x - pX) : (pX > x1 ? (pX - x1) : 0.0f);
    float dy = pY < y ? (y - pY) : (pY > y1 ? (pY - y1) : 0.0f);
    return dx*dx + dy*dy;
} // Rct::Dist2ToPt

inline void Rct::Zero()
{
    x = y = w = h = 0.0f;
}
inline void Rct::AddPoint(float _x,float _y){
    float x1=x+w;
    float y1=y+h;
    if(_x<x){
        x=_x;
    }
    if(_y<y){
        y=_y;
    }
    if(_x>x1){
        x1=_x;
    }
    if(_y>y1){
        y1=_y;
    }
    w=x1-x;
    h=y1-y;
}
inline void Rct::SetEmpty(){
    x=y=10000;
    w=h=-20000;
}
inline bool Rct::isEmpty(){
    return w<0 || h<0;
}
inline void Rct::SetPositiveDimensions()
{
    if (w < 0.0f)
    {
        x += w;
        w = -w;
    }

    if (h < 0.0f)
    {
        y += h;
        h = -h;
    }
} // Rct::SetPositiveDimensions

inline bool Rct::ClipHLine( float& px1, float& px2, float py ) const
{
    if (py < y || py >= y + h) return false;
    px1 = px1 < x ? x : px1;
    px2 = px2 < x ? x : px2;

    int vpRight  = int(x + w - 1);

    px1 = px1 > vpRight ? vpRight : px1;
    px2 = px2 > vpRight ? vpRight : px2;
    return (px1 != px2);
} // Rct::ClipHLine

inline bool Rct::IsRectInside( float px, float py, float pw, float ph ) const
{
    return (px >= x && px + pw < x + w && py >= y && py + ph < y + h);
}

inline bool Rct::ClipVLine( float px, float& py1, float& py2 ) const
{
    if (px < x || px >= x + w) return false;
    py1 = py1 < y ? y : py1;
    py2 = py2 < y ? y : py2;

    int vpBottom  = int(y + h - 1);

    py1 = py1 > vpBottom ? vpBottom : py1;
    py2 = py2 > vpBottom ? vpBottom : py2;
    return (py1 != py2);
} // Rct::ClipVLine

inline bool Rct::Clip( Rct& rct ) const
{
    int px1 = int(rct.x + rct.w);
    int py1 = int(rct.y + rct.h);

    if (rct.x < int( x ))
    {
        rct.x = x;
    }

    if (rct.y < int( y ))
    {
        rct.y = y;
    }

    int vpRight  = int(x + w);
    int vpBottom = int(y + h);
    if (px1 > vpRight)
    {
        px1 = vpRight;
    }

    if (py1 > vpBottom)
    {
        py1 = vpBottom;
    }

    rct.w = px1 - rct.x;
    rct.h = py1 - rct.y;

    if (rct.w < 0) rct.w = 0;
    if (rct.h < 0) rct.h = 0;

    return ((rct.w > 0) && (rct.h > 0));
} // Rct::ClipRect

#undef min
#undef max
inline void Rct::Union( const Rct& rct )
{
    float x0=std::min( x, rct.x );
    float y0=std::min( y, rct.y );
    float x1=std::max( x+w, rct.x+rct.w );
    float y1=std::max( y+h, rct.y+rct.h );
    x = x0;
    y = y0;
    w = x1-x0;
    h = y1-y0;
} // Rct::Union

