import os
import sys
import shutil
from pathlib import Path
import ast 
import cPy

# --- 1. Import Sphinx functions and check dependencies ---
try:
    from sphinx.cmd.build import main as sphinx_build_main
    from sphinx.ext.apidoc import main as sphinx_apidoc_main
    import pydata_sphinx_theme
except ImportError:
    print("❌ ERROR: Required packages are not installed.", file=sys.stderr)
    print("Please install them manually before running this script:", file=sys.stderr)
    print("\npip install sphinx pydata-sphinx-theme sphinx-gallery\n", file=sys.stderr)
    sys.exit(1)

# --- 2. SETTINGS: Modify these variables ---

PROJECT_NAME = "3DCoat Python API"
AUTHOR_NAME = "3DCoat"
API_MODULE_NAME = "cPy"  # The name of your API module directory

# --- End of settings ---

ROOT_DIR = Path(cPy.cCore.cExtension.getCoatInstallForder() + "/UserPrefs/PythonAPI")
DOCS_DIR = ROOT_DIR / "docs"
DOCS_SOURCE_DIR = DOCS_DIR / "source"
DOCS_BUILD_DIR = DOCS_DIR / "build"

# --- MODIFIED ---
# Це папка, що містить cPy, coat.py та CMD.py
BRIDGE_DIR = ROOT_DIR / "Bridge" 
# Це все ще шлях до cPy (використовується для перевірки імені)
API_MODULE_PATH = BRIDGE_DIR / API_MODULE_NAME 
# --- END MODIFIED ---

# This is the parent folder we add to sys.path
STD_SCRIPTS_PARENT_PATH = Path(cPy.cCore.cExtension.getCoatInstallForder() + "/UserPrefs/StdScripts")

# These are the actual packages we will scan
CMODULES_PATH = STD_SCRIPTS_PARENT_PATH / "cModules"
CTEMPLATES_PATH = STD_SCRIPTS_PARENT_PATH / "cTemplates"


def setup_directories():
    """Step 2: Set up the directory structure."""
    print(f"🧹 Cleaning and creating directory structure at {DOCS_DIR}...")
    
    if (DOCS_DIR / "build" / "html").exists():
        shutil.rmtree(DOCS_DIR / "build" / "html")
    if (DOCS_DIR / "source" / "api").exists():
        shutil.rmtree(DOCS_DIR / "source" / "api")

def create_index_rst():
    """Step 2.5: Create the main index.rst file."""
    print(f"✍️  Generating {DOCS_SOURCE_DIR / 'index.rst'}...")
    
    # This content will create the main page with a link to the API
    index_content = f"""
Welcome to {PROJECT_NAME}'s documentation!
==================================================

.. toctree::
   :maxdepth: 2
   :caption: API Reference:
   
   api/modules

Indices and tables
==================

* :ref:`genindex`
* :ref:`modindex`
* :ref:`search`
"""
    (DOCS_SOURCE_DIR / "index.rst").write_text(index_content, encoding='utf-8')
# --- END NEW FUNCTION ---

def create_conf_py():
    """Step 3: Generate the conf.py file."""
    print(f"✍️  Generating {DOCS_SOURCE_DIR / 'conf.py'}...")
    
# --- MODIFIED: Define correct paths for Sphinx ---
    bridge_path_str = str(ROOT_DIR.resolve()).replace("\\", "/")
    # Шлях до /StdScripts, щоб Sphinx міг знайти cModules, cTemplates
    std_path_str = str(STD_SCRIPTS_PARENT_PATH.resolve()).replace("\\", "/")
    
    print(f"Adding to sys.path in conf.py: {bridge_path_str}")
    print(f"Adding to sys.path in conf.py: {std_path_str}")
    # --- END MODIFIED ---

    conf_content = f"""
# conf.py
import os
import sys
import coat_type_aliases
import sphinx.ext.autodoc
from docutils import nodes
import re
import shutil
from pathlib import Path

# --- MODIFIED: Add module paths for Sphinx autodoc ---
# Це критично важливо, щоб Sphinx міг імпортувати ваші модулі
sys.path.insert(0, r"{bridge_path_str}")
sys.path.insert(0, r"{std_path_str}")
# --- END MODIFIED ---

def get_version():
    return "1.0.0"

    
# autosummary_generate = True

# -- Project information -----------------------------------------------------
project = '{PROJECT_NAME}'
copyright = '2025, {AUTHOR_NAME}'
author = '{AUTHOR_NAME}'
release = get_version()
version = '.'.join(release.split('.')[:2])

# -- General configuration ---------------------------------------------------
extensions = [
    'sphinx.ext.autodoc',      # Auto-documentation from docstrings
    'sphinx.ext.napoleon',     # Support for Google & NumPy style docstrings
    'sphinx.ext.viewcode',     # Add links to source code
    # 'sphinx.ext.autosummary',  # autosummary
]


autodoc_mock_imports = [
    "cv2",  
    "numpy",
    "PySide6",
    "PIL",
    "Coat_CPP"
]

autodoc_default_options = {{
    'undoc-members': True, 
    'show-inheritance': True,
    'show-value': True,
    'imported-members': True,
}}

#autodoc_type_aliases = coat_type_aliases.autodoc_type_aliases
add_module_names = False

templates_path = ['_templates']
exclude_patterns = []
autodoc_typehints_format = 'short'

# -- Options for HTML output (PyData Theme) ---------------------------------
html_theme = 'sphinx_book_theme'
html_static_path = ['_static']

html_css_files = [
    'custom.css',
]

html_theme_options = {{
    "logo": {{
        "text": "{PROJECT_NAME}",
    }},
    "github_url": "https://github.com/3dcarrots/3DCoat/tree/main",
    "navbar_end": ["navbar-icon-links.html", "search-field.html"],
    "show_toc_level": 2,
}}

def skip_undocumented_members(app, what, name, obj, skip, options):
    if skip:
        return True
    if what in ('module', 'attribute'):
        return False
    doc = getattr(obj, '__doc__')
    if doc and "class" in doc:
        return False
    # Changed to not skip short docstrings (e.g., from your .py stubs)
    if not doc or len(doc.strip()) < 3:
        return True
    return False

# --- :color: role function (unchanged) ---
def color_swatch_role(role, rawtext, text, lineno, inliner, options={{}}, content=[]):
    hex_code = text.strip()
    if not hex_code.startswith('#') or not (len(hex_code) == 9 or len(hex_code) == 7):
        msg = inliner.reporter.error(f"Invalid hex color code: {{text}}", line=lineno)
        prb = inliner.problematic(rawtext, rawtext, msg)
        return [prb], [msg]
    if len(hex_code) == 9: # AARRGGBB
        css_hex = f"#{{hex_code[3:]}}"
    else: # RRGGBB
        css_hex = hex_code
    swatch_style = (
        f"display: inline-block; "
        f"width: 1em; height: 1em; "
        f"border: 1px solid #888; "
        f"background-color: {{css_hex}}; "
        f"vertical-align: middle; "
        f"margin-right: 5px;"
    )
    swatch_node = nodes.raw(
        '',
        f'<span style="{{swatch_style}}"></span>',
        format='html'
    )
    text_node = nodes.Text(hex_code)
    return [swatch_node, text_node], []

def autodoc_process_docstring(app, what, name, obj, options, lines):
    # First, execute your type parsing logic
    if lines:
        first_line = lines[0].strip()
        if '(T)' in first_line: 
            parts = first_line.rsplit('(T)', 1)
            if len(parts) == 2:
                type_str = parts[0].strip()
                description_str = parts[1].strip()
                type_str = re.sub(r'\\bstatic\\b', '', type_str, flags=re.IGNORECASE)
                type_str = re.sub(r'\\bconst\\b', '', type_str, flags=re.IGNORECASE)
                type_str = type_str.strip()
                options['__my_parsed_type'] = type_str
                lines[0] = description_str
                if not description_str:
                    lines.pop(0)

    # Now, clean EVERY docstring line of "Coat_CPP."
    if lines:
        for i in range(len(lines)):
            lines[i] = re.sub(r'\\bCoat_CPP\\.', '', lines[i])

def autodoc_process_signature(app, what, name, obj, options, signature, return_annotation):
    
    # Set initial values
    sig_tuple = (signature, return_annotation)

    if options.get('__my_parsed_type'):
        parsed_type = options.pop('__my_parsed_type')
        
        if signature is None:
            signature = name
        
        original_sig = signature
        value_part = ""
        
        if ' = ' in original_sig:
            parts = original_sig.split(' = ', 1)
            attr_name = parts[0].strip()
            value_part = " = " + parts[1].strip()
        else:
            attr_name = original_sig.strip()
        
        if value_part.startswith(" = <object"):
            value_part = ""
        
        if ' : object' in attr_name:
            attr_name = attr_name.split(' : object', 1)[0].strip()

        new_signature = f"{{attr_name}} : {{parsed_type}}{{value_part}}"
        sig_tuple = (new_signature, None) # Set new signature
            
    else:
        options.pop('__my_parsed_type', None)

    # Now, clean the final signature and return annotation
    final_sig, final_ret = sig_tuple
    
    if final_sig:
        final_sig = re.sub(r'\\bCoat_CPP\\.', '', final_sig)
    if final_ret:
        final_ret = re.sub(r'\\bCoat_CPP\\.', '', final_ret)
        
    return (final_sig, final_ret)


class ColorAttributeDocumenter(sphinx.ext.autodoc.AttributeDocumenter):
    priority = 11

    # (No longer needed, 'autodoc_mock_imports' will fix everything)

    # (This is needed to return values and colors)
    def add_directive_header(self, sig: str) -> None:
        super(sphinx.ext.autodoc.AttributeDocumenter, self).add_directive_header(sig)

        if not self.options.get('show-value', True):
            return
            
        try:
            if (self.object is sphinx.ext.autodoc.INSTANCEATTR or 
               (repr(self.object).startswith("<") and " object at " in repr(self.object))):
                return
        except Exception:
            pass

        is_color_name = 'Color' in self.objpath[-1]
        is_int_type = isinstance(self.object, int)

        self.add_line('', self.get_sourcename())

        if is_color_name and is_int_type:
            try:
                val = int(self.object)
                hex_code = f"#{{val & 0xFFFFFFFF:08X}}"
                self.add_line(f'   :value: :color:`{{hex_code}}`', self.get_sourcename())
            except Exception:
                val_repr = repr(self.object)
                val_repr = re.sub(r'\\bCoat_CPP\\.', '', val_repr)
                self.add_line('   :value: ' + val_repr, self.get_sourcename())
        else:
            val_repr = repr(self.object)
            val_repr = re.sub(r'\\bCoat_CPP\\.', '', val_repr)
            self.add_line('   :value: ' + val_repr, self.get_sourcename())

def copy_doxygen_assets(app, exception):
    if exception:
        return 

    src_dir = Path(app.srcdir) / "tutorials"
    out_dir = Path(app.outdir) / "tutorials"

    if not src_dir.exists():
        print(f"Doxygen source dir {{src_dir}} not found. Skipping asset copy.")
        return

    shutil.copytree(
        src_dir,
        out_dir,
        ignore=shutil.ignore_patterns("*.rst", "*.html"),
        dirs_exist_ok=True
    )
    print(f"Copied Doxygen assets from {{src_dir}} to {{out_dir}}")

def setup(app):
    app.connect('build-finished', copy_doxygen_assets)
    app.connect('autodoc-skip-member', skip_undocumented_members)
    app.connect('autodoc-process-docstring', autodoc_process_docstring)
    app.connect('autodoc-process-signature', autodoc_process_signature)
    app.add_role("color", color_swatch_role)
    app.add_autodocumenter(ColorAttributeDocumenter)
"""
    (DOCS_SOURCE_DIR / "conf.py").write_text(conf_content, encoding='utf-8')

def cleanup_rst_files(directory, find_text, replace_text):
    if not os.path.exists(directory):
        print(f"ERROR: {directory} not found.")
        return

    file_count = 0
    for root, dirs, files in os.walk(directory):
        for filename in files:
            if filename.endswith('.rst'):
                file_path = os.path.join(root, filename)
                try:
                    with open(file_path, 'r', encoding='utf-8') as f:
                        lines = f.readlines()
                    
                    if not lines:
                        continue

                    first_line = lines[0]
                    if find_text in first_line:
                        lines[0] = first_line.replace(find_text, replace_text)
                        
                        if len(lines) > 1:
                            underline_char = lines[1].strip()[0:1] 
                            if underline_char in ('=', '-', '`', ':', '.', "'", '"', '~', '^', '_', '*'):
                                new_title_len = len(lines[0].rstrip())
                                lines[1] = underline_char * new_title_len + '\n'

                        with open(file_path, 'w', encoding='utf-8') as f:
                            f.writelines(lines)
                        
                        file_count += 1
                
                except Exception as e:
                    print(f"ERROR {file_path}: {e}")
                    


def add_members_to_rst(directory):
    """
    Finds all .rst files and adds the :members: option if it's missing.
    """
    print(f"\n--- 3. Adding :members: to RST files in {directory} ---")
    target_text = ":show-inheritance:"
    replacement_text = ":show-inheritance:\n   :members:" 
    file_count = 0

    for root, dirs, files in os.walk(directory):
        for filename in files:
            if filename.endswith('.rst'):
                file_path = os.path.join(root, filename)
                try:
                    with open(file_path, 'r', encoding='utf-8') as f:
                        content = f.read()
                    
                    made_change = False
                    
                    if target_text in content and ":members:" not in content:
                        content = content.replace(target_text, replacement_text)
                        made_change = True
                        
                    if made_change:
                        with open(file_path, 'w', encoding='utf-8') as f:
                            f.write(content)
                        file_count += 1
                
                except Exception as e:
                    print(f"Error processing file {file_path}: {e}")
                    
    print(f"--- 3.1. Added :members: to {file_count} files. ---")

def run_sphinx_commands():
    """Steps 4, 5, & 6: Run sphinx-apidoc, create index, and run sphinx-build."""
    
    # Step 4: Call sphinx-apidoc directly (THREE TIMES)
    print("🤖 Running sphinx-apidoc to scan API...")
    api_output_dir = DOCS_SOURCE_DIR / "api"
    
# --- Call 1: Scanning 'Bridge' directory (for cPy, coat, CMD) ---
    apidoc_args_1 = [
        "-o", str(api_output_dir),
        str(BRIDGE_DIR), # /PythonAPI/Bridge
        "--force",
        "--separate",
        "--no-toc",
    ]
        
    # --- Call 2: Scanning 'cModules' ---
    apidoc_args_2 = [
        "-o", str(api_output_dir),
        str(CMODULES_PATH), # /StdScripts/cModules
        "--force",
        "--separate",
        "--no-toc",
    ]
    
    # --- Call 3: Scanning 'cTemplates' ---
    apidoc_args_3 = [
        "-o", str(api_output_dir),
        str(CTEMPLATES_PATH), # /StdScripts/cTemplates
        "--force",
        "--separate",
        "--no-toc",
    ]
    
    try:
        print(f"--- Scanning {BRIDGE_DIR}...")
        sphinx_apidoc_main(apidoc_args_1)        
        
        print(f"--- Scanning {CMODULES_PATH}...")
        sphinx_apidoc_main(apidoc_args_2)

        print(f"--- Scanning {CTEMPLATES_PATH}...")
        sphinx_apidoc_main(apidoc_args_3)
        
    except Exception as e:
        print(f"❌ ERROR during sphinx-apidoc: {e}", file=sys.stderr)
        sys.exit(1)

    # --- STEP 5: Filter and generate 'modules.rst' ---
    
    all_rst_stems = sorted([f.stem for f in api_output_dir.glob("*.rst") if f.name != "modules.rst"])
    documented_modules = []

    print(f"🔍 Analyzing {len(all_rst_stems)} found modules for documentation...")

    for stem in all_rst_stems:
        source_file_path = None
        module_path_parts = stem.split('.')
        
        # Determine where to look for the source .py file
        if module_path_parts[0] == API_MODULE_NAME: # "Bridge"
            base_path = API_MODULE_PATH
            relative_path_parts = module_path_parts[1:]
        elif module_path_parts[0] == CMODULES_PATH.name: # "cModules"
            base_path = CMODULES_PATH
            relative_path_parts = module_path_parts[1:]
        elif module_path_parts[0] == CTEMPLATES_PATH.name: # "cTemplates"
            base_path = CTEMPLATES_PATH
            relative_path_parts = module_path_parts[1:]
        else:
            # This might be a file in the root of /StdScripts, e.g., 'some_script.py'
            base_path = STD_SCRIPTS_PARENT_PATH
            relative_path_parts = module_path_parts

        # Find the path to .py or __init__.py
        if not relative_path_parts: # This is top-level, e.g., Bridge.rst or cModules.rst
            test_path_init = base_path / '__init__.py'
            if test_path_init.exists():
                source_file_path = test_path_init
            else:
                # This is a .py file in the StdScripts root, e.g., some_script.py
                test_path_py = base_path.with_suffix('.py')
                if test_path_py.exists():
                    source_file_path = test_path_py
        else: # This is a sub-module, e.g., Bridge.cPy.rst
            test_path_py = base_path / Path(*relative_path_parts).with_suffix('.py')
            test_path_init = base_path / Path(*relative_path_parts) / '__init__.py'
            
            if test_path_py.exists():
                source_file_path = test_path_py
            elif test_path_init.exists():
                source_file_path = test_path_init

        is_documented = False
        if source_file_path and source_file_path.exists():
            try:
                content = source_file_path.read_text(encoding='utf-8')
                if not content.strip(): # File is empty
                    is_documented = False
                else:
                    tree = ast.parse(content)
                    module_docstring = ast.get_docstring(tree)
                    
                    if module_docstring and module_docstring.strip():
                        # 1. The module has a docstring
                        is_documented = True
                    else:
                        for node in tree.body:
                            if isinstance(node, (ast.FunctionDef, ast.AsyncFunctionDef, ast.ClassDef)):
                                # 2. The module has classes or functions
                                is_documented = True
                                break
                            if isinstance(node, ast.Assign):
                                for target in node.targets:
                                    if isinstance(target, ast.Name) and target.id == '__all__':
                                        # 3. The module defines __all__ (like your coat.py)
                                        is_documented = True
                                        break
                                if is_documented: break
            except Exception as e:
                print(f"  [Warning] Could not parse {source_file_path}. Including it. Error: {e}")
                is_documented = True # Include if parsing failed
        else:
            # This is a C++ module (e.g., 'coat') or an .rst file without a .py (e.g., 'modules')
            # Or a .pyd we "mocked"
            # Include it since we can't check
            if stem not in ("modules"):
                is_documented = True 

        if is_documented:
            documented_modules.append(stem)
        else:
            print(f"  -> Skipping {stem} (no docstring, functions, classes, or __all__ found)")
# --- End of filtering logic ---

    print(f"✍️  Generating main API file with {len(documented_modules)} modules: {api_output_dir / 'modules.rst'}...")

    # --- НОВЕ: Сортуємо модулі по групах ---
    # (Ми додаємо `or m == ...` щоб включити самі головні файли cPy.rst, cModules.rst, і т.д.)
    BRIDGE_PACKAGE_NAME = BRIDGE_DIR.name 
    
    cpy_modules = sorted([m for m in documented_modules if 
                              m.startswith(f"{BRIDGE_PACKAGE_NAME}.") or # Bridge.submodule
                              m == BRIDGE_PACKAGE_NAME                  # Bridge (the package itself)
                             ])
    # --- END MODIFIED ---
    cmodules_modules = sorted([m for m in documented_modules if m.startswith(f"{CMODULES_PATH.name}.") or m == CMODULES_PATH.name])

    ctemplates_modules = sorted([m for m in documented_modules if m.startswith(f"{CTEMPLATES_PATH.name}.") or m == CTEMPLATES_PATH.name])
    
    # Знаходимо всі інші модулі, які не потрапили в групи
    all_grouped_modules = set(cpy_modules) | set(cmodules_modules) | set(ctemplates_modules)
    other_modules = sorted([m for m in documented_modules if m not in all_grouped_modules])
    # --- Кінець НОВОГО ---

    # --- НОВЕ: Будуємо modules_content з окремими toctree ---
    modules_content = "API Modules\n=========\n\n"

    if not documented_modules:
        modules_content += "(No documented modules found.)"
    else:
        # Група 1: cPy API
        if cpy_modules:
            # :caption: - це те, що створить окрему гілку меню
            modules_content += ".. toctree::\n   :maxdepth: 4\n   :caption: cPy API (Bridge)\n\n"
            for module in cpy_modules:
                modules_content += f"   {module}\n"
            modules_content += "\n"

        # Група 2: cModules API
        if cmodules_modules:
            modules_content += ".. toctree::\n   :maxdepth: 4\n   :caption: cModules API\n\n"
            for module in cmodules_modules:
                modules_content += f"   {module}\n"
            modules_content += "\n"

        # Група 3: cTemplates API
        if ctemplates_modules:
            modules_content += ".. toctree::\n   :maxdepth: 4\n   :caption: cTemplates API\n\n"
            for module in ctemplates_modules:
                modules_content += f"   {module}\n"
            modules_content += "\n"
            
        # Група 4: Інші модулі (якщо є)
        if other_modules:
            modules_content += ".. toctree::\n   :maxdepth: 4\n   :caption: Other Modules\n\n"
            for module in other_modules:
                modules_content += f"   {module}\n"
            modules_content += "\n"
    
    (api_output_dir / "modules.rst").write_text(modules_content, encoding='utf-8')
    # --- Кінець НОВОГО ---
    
    # --- End of step 5 ---
    add_members_to_rst(api_output_dir)
  
    # Clean up 'Bridge.' and 'coat.' (if present)
    cleanup_rst_files(DOCS_DIR / "source" / "api", "Bridge.", "")
    cleanup_rst_files(DOCS_DIR / "source" / "api", "coat.", "")
    
    # Step 6: Call sphinx-build directly
    print("🛠️  Running sphinx-build to generate HTML...")
    html_output_dir = DOCS_BUILD_DIR / "html"
    
    build_args = [
        "-b", "html",
        str(DOCS_SOURCE_DIR),
        str(html_output_dir)
    ]
    
    return_code = sphinx_build_main(build_args)
    
    if return_code != 0:
        print(f"❌ Sphinx build failed with exit code {return_code}", file=sys.stderr)
        sys.exit(return_code)
        
def main():
    """Main script execution function."""
    print(f"--- Starting documentation build for {PROJECT_NAME} (Native Python) ---")
    
    # 2. Setup directories
    setup_directories()

    # 2.5 Create the main index.rst
    # create_index_rst()
    
    # 3. Create conf.py
    create_conf_py()
    
    # 4, 5, 6. Run Sphinx
    run_sphinx_commands()
    
    # 7. Finish
    final_path = (DOCS_BUILD_DIR / "html" / "index.html").resolve()
    print("\n" + "="*50)
    print("🎉 DOCUMENTATION BUILT SUCCESSFULLY! 🎉")
    print(f"Open this file in your browser:\nfile://{final_path}")
    print("="*50)