#pragma once
enum FieldClamp {
	fcWrap = 0,
	fcClamp=1,
	fcZero=2
};

/**
 * \brief The class to operate over the 2D fields
 */
class APICALL ValuesField {
	int Lx, Ly, Depth;
	cList<float> values;
	FieldClamp clamp;
	cList<ValuesField*> mips;
public:
	ValuesField();
	~ValuesField();

	/**
	 * \brief construct the field from the other cropped field
	 * \param vf the field to copy
	 * \param x the x-coordinate of the left-top corner (of the vf)
	 * \param y the y-coordinate of the left-top corner (of the vf)
	 * \param w the width of the field, if zero the whole field will be copied (shifted on x)
	 * \param h the height of the field, if zero the whole field will be copied (shifted on y)
	 */
	ValuesField(const ValuesField& vf, int x = 0, int y = 0, int w = 0, int h = 0);

	/**
	 * \brief Create the field
	 * \param SizeX width
	 * \param SiezY height
	 * \param depth channels count
	 * \param clampmode clamp mode
	 */
	void  create(int SizeX, int SiezY, int depth = 1, FieldClamp clampmode = fcWrap);

	/**
	 * \brief create the mipmaps, decreasing twice with each level until it is possible
	 */
	void createMips();

	/**
	 * \brief set the field clamp mode - how it acts beyond the boundaries
	 * \param cl the clamp mode - fcWrap, fcClamp or fcZero
	 */
	void  setClampMode(FieldClamp cl);

	/**
	 * \brief get width
	 * \return width of the field
	 */
	int   width() const ;

	/**
	 * \brief get height
	 * \return height of the field
	 */
	int   height() const ;

	/**
	 * \brief Get amount of channels
	 * \return channels count
	 */
	int   depth() const ;

	/**
	 * \brief create the field form the image
	 * \param im the image to be converted to the float field
	 */
	void  createFromImage(const comms::cImage& im);

	/**
	 * \brief create the monochromatic field from the image (1 channel)
	 * \param im the image, if it is RGB, the channels will be merged to greyscale
	 * \param mask the mask to blend rgb - channels, if zero - the default rgb mask will be used
	 */
	void  createFromImageMono(const comms::cImage& im, Vector3D mask = Vector3D::Zero);

	/**
	 * \brief convert the field to the image
	 * \param im destination image, values should be 0..255 fro 3 or 4 cannels field, if there is 1 channel, the values will be scaled from minimum to maximum in range 0..255
	 */
	void  toImage(comms::cImage& im);

	/**
	 * \brief read the image directly from the file
	 * \param name the filename
	 */
	void  loadImage(const char* name);

	/**
	 * \brief save the field directly to file
	 * \param name the filename
	 */
	void  saveImage(const char* name);

	/**
	 * \brief clear rthe field
	 */
	void  clear();

	/**
	 * \brief converts the (x,y) to range [0..Lx-1, 0..Ly-1]
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \return true if succeed and wrapping happened
	 */
	bool  torange(int& x, int& y) const;

	/**
	 * \brief set teh value to the field
	 * \param value the value to set
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 */
	void  set(float value, int x, int y, int channel = 0);

	/**
	 * \brief add the value to the field
	 * \param value the value to add
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 */
	void  add(float value, int x, int y, int channel = 0);

	/**
	 * \brief get the value of the field by the integer coordinates
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the value of the field
	 */
	float get(int x, int y, int channel = 0) const ;

	/**
	  * \brief get the value reference of the field by the integer coordinates
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the value of the field
	 */
	float& value(int x, int y, int channel = 0);

	/**
	 * \brief get the reference to the values as 3d - vector. It has sense only if there are at least 3 channels
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \return the reference to the vector
	 */
	Vector3D& getV3(int x, int y);
	Vector3D getV3(int x, int y) const;

	/**
	 * \brief get the reference to the values as 4d - vector. It has sense only if there are at least 4 channels
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \return the reference to the 4d-vector
	 */
	Vector4D& getV4(int x, int y);
	Vector4D getV4(int x, int y) const;

	/**
	 * \brief get the reference to the values as 2d - vector. It has sense only if there are at least 2 channels
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \return the reference to the vector
	 */
	cVec2& getV2(int x, int y);
	cVec2 getV2(int x, int y) const;

	/**
	 * \brief get the linearly interpolated value
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated value reference
	 */
	float get_linear(float x, float y, int channel = 0) const ;

	/**
	 * \brief get the linearly interpolated value using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param channel the channel
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	float get_linear_mip(float x, float y, int channel, float mip_level) const;

	/**
	 * \brief get the linearly interpolated value as 2d-vector
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated 2d-value
	 */
	cVec2 get_linearV2(float x, float y) const;

	/**
	 * \brief get the linearly interpolated value as 2d-vector using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	cVec2 get_linearV2_mip(float x, float y, float mip_level) const;

	/**
	 * \brief get the linearly interpolated value as 3d-vector
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated 2d-value
	 */
	cVec3 get_linearV3(float x, float y) const;

	/**
	 * \brief get the linearly interpolated value as 3d-vector using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	cVec3 get_linearV3_mip(float x, float y, float mip_level) const;

	/**
	 * \brief get the linearly interpolated value as 4d-vector
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated 2d-value
	 */
	cVec4 get_linearV4(float x, float y) const;

	/**
	 * \brief get the linearly interpolated value as 4d-vector using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param mip_level the mip level
	 * \return the interpolated value (assuming that w is the alpha channel)
	 */
	cVec4 get_linearV4_mip(float x, float y, float mip_level) const;

	/**
	 * \brief get the bicubic interpolated value
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated value
	 */
	float get_bicubic(float x, float y, int channel = 0) const ;

	/**
	 * \brief get the bicubic interpolated value using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param channel the channel
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	float get_bicubic_mip(float x, float y, int channel, float mip_level) const;

	/**
	 * \brief get the bicubic interpolated value as the 2d-vector
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated value
	 */
	cVec2 get_bicubicV2(float x, float y) const;

	/**
	 * \brief get the bicubic interpolated value as the 2d-vector using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	cVec2 get_bicubicV2_mip(float x, float y, float mip_level) const;

	/**
	 * \brief get the bicubic interpolated value as the 3d-vector
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated value
	 */
	cVec3 get_bicubicV3(float x, float y) const;

	/**
	 * \brief get the bicubic interpolated value as the 3d-vector using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	cVec3 get_bicubicV3_mip(float x, float y, float mip_level) const;

	/**
	 * \brief get the bicubic interpolated value as the 4d-vector
	 * \param x x-coordinate
	 * \param y y-coordinate
	 * \param channel the channel
	 * \return the interpolated value
	 */
	cVec4 get_bicubicV4(float x, float y) const;

	/**
	 * \brief get the bicubic interpolated value as the 4d-vector using the mipmaps
	 * \param x the x-coordinate
	 * \param y the y-coordinate
	 * \param mip_level the mip level
	 * \return the interpolated value
	 */
	cVec4 get_bicubicV4_mip(float x, float y, float mip_level) const;

	/**
	 * \brief resize the field to other size using the bicubic interpolation
	 * \param dest the destination field
	 */
	void  resizeTo(ValuesField& dest);

	/**
	 * \brief Experimental: Increase the image 2X and sharpen 
	 * \param dest the destination, should be empty
	 */
	void  sharp2X(ValuesField& dest);

	/**
	 * \brief Experimental: Increase the image 2X and sharpen (sharpening edges so that edge width will be kept despite on the image size increasing)
	 * \param dest the destination, should be empty
	 */
	void  sharpEdges2X(ValuesField& dest);

	/**
	 * \brief Experimental, variation of the previous function: Increase the image 2X and sharpen (sharpening edges so that edge width will be kept despite on the image size increasing)
	 * \param dest the destination, should be empty
	 */
	void  sharpEdges2X1(ValuesField& dest);

	/**
	 * \brief resize to bigger size keeping the edges sharpness
	 * \param dest destination image
	 * \param scale the scale > 1
	 */
	void  sharpResize(ValuesField& dest, float scale);

	/**
	 * \brief relax the field
	 * \param destination the destination field. It is recommended to use swap for multiple relax steps
	 * \param degree the degree of the relaxation
	 * \param radius the radius of the kernel
	 */
	void  relax(ValuesField& destination, float degree, float radius);
	void  relax(ValuesField& destination, float degree, float radius, int x0, int y0, int x1, int y1);


	/**
	 * \brief sharpen the field
	 * \param destination the destination field. It is recommended to use swap for multiple relax steps
	 * \param degree the degree of the relaxation
	 * \param radius the radius of the kernel
	 */
	void sharpen(ValuesField& destination, float degree, float radius);

	/**
	 * \brief convert multiple channels to one as an average-arithmetic
	 * \param dest 
	 */
	void  monoTo(ValuesField& dest);

	/**
	 * \brief calculate the curvature of the 2d-field
	 * \param dest the destination field (1 channel)
	 */
	void  curvature(ValuesField& dest);

	/**
	 * \brief calculate the curvature principal directions
	 * \param dest the destination field (3channels)
	 */
	void  curvatureDir(ValuesField& dest);

	/**
	 * \brief calculate the normalmap
	 * \param dest the destination field (2 channels, only dx and dy will be written)
	 * \param byChannel the channel to be used as the depth source
	 */
	void  normalmap(ValuesField& dest, int byChannel = 0);

	/**
	 * \brief calculate the normalmap as the 3d-vectors set
	 * \param dest the destination 3d-field
	 * \param heightmod the modulator for the heightmap
	 * \param channel the channel to be used as the depth source
	 */
	void  normalmap3(ValuesField& dest, float heightmod = 1.0, int channel = 0);

	/**
	 * \brief calculate the gradient length
	 * \param dest the 1d field
	 */
	void  gradientValue(ValuesField& dest);

	//experimental functions...
	void  perlinMap(ValuesField& dest);
	void  detectAnisotropy(ValuesField& dest);
	float detectPerlin(int x, int y, int channel);

	/**
	 * \brief subtract field from the current
	 * \param field the subtracted field, same dimensions required
	 */
	void operator -= (const ValuesField& field);

	/**
	 * \brief add field to the current
	 * \param field the field to add (same dimensions)
	 */
	void operator += (const ValuesField& field);

	/**
	 * \brief add the constant value to the field
	 * \param value the value to add
	 */
	void operator += (float value);

	/**
	 * \brief subtract the value from the field
	 * \param value the value to subtract
	 */
	void operator -= (float value);

	/**
	 * \brief multiply on the constant
	 * \param value the multiplicator
	 */
	void operator *= (float value);

	/**
	 * \brief divide on the constant
	 * \param value the divider
	 */
	void operator /= (float value);

	/**
	 * \brief operate over the field with the lambda
	 * \param f the lambda
	 */
	void operate(const std::function<void(int, int) >& f, bool multithread = true);

	/**
	 * \brief calculated the maximum of two fields into hte current
	 * \param other the other field
	 */
	void max(ValuesField& other);

	/**
	 * \brief swap two fields (dimensions may be different)
	 * \param dest the other field
	 */
	void swap(ValuesField& dest);

	/**
	 * \brief Interpolates the whole field so that it is maximally smooth (laplacian value minimized) and gets the given values at the each line provided,
	 * so that field_value(point.x,point.y) = point.z. This is heavy function, be careful with the timing. But it produces absolutely clean and smooth field.
	 * \param lines the pairs of points that define the field, x,y of each point may be not integer, the count of points should be even
	 */
	void LaplacianInterpolate(const cList<Vector3D>& lines);

	/**
	 * \brief Interpolates the whole field so that it is maximally smooth (laplacian value minimized) and keeps the grid value at each locked point.
	 * This is heavy function, be careful with the timing. But it produces absolutely clean and smooth field.
	 * \param fixed The fixel points falgs, fixed.get(x+y*width) == true means that point is fixed
	 */
	void LaplacianInterpolate(const UnlimitedBitset& fixed);

	/**
	 * \brief Normalize the texture to be (128, 128, 128) in average using the mip-mapping algorithm
	 * \param monochromatic set true to change the brightness only
	 * \param mipsize the mip-map size to be used to normalize the color
	 */
	void NormalizeColors(bool monochromatic = false, int mipsize = 64);

	/**
	 * \brief get the average color of the texture piece
	 * \param x the x-coordinate of the piece
	 * \param y the y-coordinate of the piece
	 * \param w the width of the piece, iw w == 0, the whole texture width is taken
	 * \param h the height of the piece, if h == 0, the whole texture height is taken
	 * \param alpha_threshold the threshold for the alpha channel (if exists)
	 * \return the average color, taken only the color of pixels with alpha > alpha_threshold
	 */
	comms::cVec3 averageColor(int x = 0, int y = 0, int w = 0, int h = 0, float alpha_threshold = 200) const;

	/**
	 * \brief Decrease the field 2X times
	 * \param dest the destination field
	 */
	void Decrease2X(ValuesField& dest);

	/**
	 * \brief resize to the closest power of 2 (may increase, may decrease the size, we choose what is arithmetically closer)
	 * \param dest the destination
	 */
	void toPower2(ValuesField& dest);

	/**
	 * \brief recover the bump from the color/greyscale (heuristic)
	 * \param dest the destination depthmap
	 */
	void recoverBump(ValuesField& dest);

	/**
	 * \brief blend the image with the other image
	 * \param other the other image to blend with this image, it should be of the 4-channels in depth (color + alpha)
	 * \param transform the transformation matrix applied to the other image
	 * \param modulator the color/alpha modulator
	 */
	comms::cBounds blendImage(const ValuesField& other, const cMat3& transform, const cVec4& modulator = cVec4::One);

	/**
	 * \brief blend the image with the other image using the 4x4 matrix transformation
	 * \param other the other image to blend with this image, it should be of the 4-channels in depth (color + alpha)
	 * \param transform the transformation matrix applied to the other image
	 * \param modulator the color/alpha modulator
	 * \return the bounds of the blended image
	 */
	comms::cBounds blendImageM4(const ValuesField& other, const comms::cMat4& transform, const cVec4& modulator = cVec4::One);

	/**
	 * \brief draw the filled rectangle
	 * \param x0 the left-top corner x
	 * \param y0 the left-top corner y
	 * \param x1 the right-bottom corner x
	 * \param y1 the right-bottom corner y
	 * \param color the color of the rectangle (r, g, b a), all values are in [0..255]
	 */
	void rect(int x0, int y0, int x1, int y1, const cVec4& color);

	void calc_featuremap(ValuesField& dest, int radius);

	void local_normalisation(ValuesField& dest, int size);

	void find_local_minmax(std::vector<comms::cVec2>& minmax);

	void getEllipse(int x, int y, cVec2& center, cVec2& longer_axis, float& relation);
};

inline int   ValuesField::width() const {
	return Lx;
}

inline int   ValuesField::height() const {
	return Ly;
}

inline int	 ValuesField::depth() const {
	return Depth;	
}

