#pragma once

//*****************************************************************************
// cDimension
//*****************************************************************************
struct cDimension {
	enum Enum {
		None	= 0,
		OneD	= 1,
		TwoD	= 2,
		ThreeD	= 3,
		Cube	= 4
	};

	static const char * ToString(const Enum Dimension);
	static const Enum ToDimension(const int Width, const int Height, const int Depth) {
		if(1 == Depth && 1 == Height) {
			return OneD;
		} else if(1 == Depth && Height > 1) {
			return TwoD;
		} else if(Depth > 1) {
			return ThreeD;
		} else if(0 == Depth) {
			return Cube;
		}
		return None;
	}
}; // cDimension

//*****************************************************************************
// cFormat
//*****************************************************************************
struct cFormat {
	enum Enum {
		None	= 0,
		
		// Plain:
		R8		= 1,
		Rg8		= 2,
		Rgb8	= 3,
		Rgba8	= 4,
		
		R16		= 5,
		Rg16	= 6,
		Rgb16	= 7,
		Rgba16	= 8,
		
		R16f	= 9,
		Rg16f	= 10,
		Rgb16f	= 11,
		Rgba16f = 12,
		
		R32f	= 13,
		Rg32f	= 14,
		Rgb32f	= 15,
		Rgba32f	= 16,

		// Depth:
		Depth16			= 17,
		Depth24			= 18,
		Depth24Stencil8 = 19,
		
		// Compressed:
		Dxt1	= 20,
		Dxt3	= 21,
		Dxt5	= 22,
        
        // PowerVR Texture Compression
        PVRTC4			= 23,
        PVRTC4_Alpha	= 24,
		Count			= 25
	};

	static bool IsPlain(const Enum Format) {
		return Format <= Rgba32f;
	}

	static bool IsDepth(const Enum Format) {
		return Format >= Depth16 && Format <= Depth24Stencil8;
	}

	static bool HasStencil(const Enum Format) {
		return Depth24Stencil8 == Format;
	}

	static bool IsCompressed(const Enum Format) {
		return Format >= Dxt1 && Format <= PVRTC4_Alpha;
	}

	static bool IsFloat(const Enum Format) {
		return Format >= R16f && Format <= Rgba32f;
	}
    
	static int ChannelCount(const Enum Format) {
		static const int Count[] = {
			0, // None
			1, // R8
			2, // Rg8
			3, // Rgb8
			4, // Rgba8
			1, // R16
			2, // Rg16
			3, // Rgb16
			4, // Rgba16
			1, // R16f
			2, // Rg16f
			3, // Rgb16f
			4, // Rgba16f
			1, // R32f
			2, // Rg32f
			3, // Rgb32f
			4, // Rgba32f
			1, // Depth16
			1, // Depth24
			1, // Depth24Stencil8
			3, // Dxt1
			4, // Dxt3
			4, // Dxt5
			3, // PVRTC4
			4  // PVRTC4_Alpha
		};
		return Count[Format];
	}

	static int BytesPerChannel(const Enum Format) { // Only plain formats
		cAssert(IsPlain(Format));

		if(Format <= Rgba8) {
			return 1;
		}
		if(Format <= Rgba16f) {
			return 2;
		}
		return 4;
	}

	static int BytesPerPixel(const Enum Format) { // Only plain formats
		cAssert(IsPlain(Format));
		
		static const int Bytes[] = {
			0, // None
			1, // R8
			2, // Rg8
			3, // Rgb8
			4, // Rgba8
			2, // R16
			4, // Rg16
			6, // Rgb16
			8, // Rgba16
			2, // R16f
			4, // Rg16f
			6, // Rgb16f
			8, // Rgba16f
			4, // R32f
			8, // Rg32f
			12, // Rgb32f
			16, // Rgba32f
			-1, // Depth16
			-1, // Depth24
			-1, // Depth24Stencil8
			-1, // Dxt1
			-1, // Dxt3
			-1, // Dxt5
			-1, // PVRTC4
			-1  // PVRTC4_Alpha
		};
		return Bytes[Format];
	}

	static int BytesPerBlock(const Enum Format) { // Only compressed formats
		cAssert(IsCompressed(Format));
        return (Dxt5 == Format || Dxt3 == Format) ? 16 : 8;
	}

	static const char * ToString(const Enum Format);
	static Enum ToFormat(const char *String);
}; // cFormat

//*****************************************************************************
// cImage
//*****************************************************************************
class APICALL cImage {
	friend class cCodecDds;
	friend class cCodecJpeg;
	friend class cCodecTga;
	friend class cCodecPng;
	friend class cCodecTiff;
public:
    static int CalcImageSize(const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);

	enum Constants {
		MipMapsAll = 127
	};

	cImage();

	cImage(const cImage &Src);
	void Copy(const cImage &Src);

	cImage(const cImage &Src, const cRect &rc);
	void Copy(const cImage &Src, const cRect &rc);
	
	enum ECopyCtor { CopyCtor };
	enum ESetCtor { SetCtor };

	void Copy(const void *Pixels, const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);
	cImage(const ECopyCtor, const void *Pixels, const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);

	void Set(void *Pixels, const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);
	void SetHeight(int Height);
	cImage(const ESetCtor, void *Pixels, const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);

	void Create(const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);
	cImage(const cFormat::Enum Format, const int Width, const int Height, const int Depth, const int MipMapCount);

	void Free();
	~cImage();
	
	int GetWidth() const {
		return m_Width;
	}
	int GetHeight() const {
		return m_Height;
	}
	int GetDepth() const {
		return m_Depth;
	}
	int GetWidth(const int MipMapLevel) const {
		return cMath::Max(1, m_Width >> MipMapLevel);
	}
	int GetHeight(const int MipMapLevel) const {
		return cMath::Max(1, m_Height >> MipMapLevel);
	}
	int GetDepth(const int MipMapLevel) const {
		return cMath::Max(1, m_Depth >> MipMapLevel);
	}
	
	cFormat::Enum GetFormat() const {
		return m_Format;
	}
	
    int GetSize() const {
        return CalcImageSize(m_Format, m_Width, m_Height, m_Depth, m_MipMapCount);
    }
	byte * GetPixels() const {
		return m_Pixels;
	}
	byte * GetPixels(const int MipMapLevel) const;
	
	int GetMipMapCount() const { return m_MipMapCount; }
	int GetMipMapCountFromDimensions() const;
	int GetMipMappedSize(const int MipMapFrom = 0, const int MipMapCount = MipMapsAll, const cFormat::Enum SrcFormat = cFormat::None) const;
	int GetSliceSize(const int MipMapLevel = 0, const cFormat::Enum SrcFormat = cFormat::None) const;
	int GetRowSize(const int MipMapLevel = 0, const cFormat::Enum SrcFormat = cFormat::None) const;
	int GetPixelCount(const int MipMapFrom = 0, const int MipMapCount = MipMapsAll) const;
	
	const cDimension::Enum GetDimension() const {
		return cDimension::ToDimension(m_Width, m_Height, m_Depth);
	}
	
	// Only plain formats:
	bool SwapChannels(const int Ch0, const int Ch1);
	bool RemoveChannels(bool KeepCh0, bool KeepCh1 = true, bool KeepCh2 = true, bool KeepCh3 = true);
	bool InvertChannel(const int Ch);
	bool CopyChannel(const int From, const int To);

	bool ToFormat(const cFormat::Enum Format);
	
	// Dxt1 -> Rgb8
	// Dxt3, Dxt5 -> Rgba8
	void Uncompress();

	// R8, Rgb8 -> Dxt1
	// Rgba8 -> Dxt5
	void Compress();
	
	// Rgb8, Rgba8 -> R8
	// Rgb16, Rgba16 -> R16
	bool ToGrayScale();
	
	// From: R8, Rgb8, Rgba8	
	// To (2D): Rg8, Rgba8
	// To (3D, Volume Texture): Rgb8, Rgba8
	// "OldAlpha" works only if(Rgba8 == m_Format && Rgba8 == Format)
	bool ToNormalMap(const cFormat::Enum Format, const float Z = 1.0f, const bool OldAlpha = false);

	bool CreateMipMaps(const int MipMapCount = MipMapsAll); // !Compressed
	void RemoveMipMaps();
	
	//*************************************************************************
	// GetPixel(Rgb8, Rgba8, R8)
	//*************************************************************************
	struct PixelRgb8 {
		byte r, g, b;
	};
	struct PixelRgba8 {
		byte r, g, b, a;
	};
	const PixelRgb8 GetPixelRgb8(const int X, const int Y) const;
	void SetPixelRgb8(const int X, const int Y, const PixelRgb8 &p);

	const PixelRgba8 GetPixelRgba8(const int X, const int Y) const;
	void SetPixelRgba8(const int X, const int Y, const PixelRgba8 &p);
	
	byte GetPixelR8(const int X, const int Y) const;
	void SetPixelR8(const int X, const int Y, const byte p);

	void MergeRgb8(const cImage &From, const int X, const int Y);
	void MergeRgba8(const cImage& From, const int X, const int Y, bool ColorAsAlpha);

	bool Flip(); // Plain, packed and compressed formats
	bool Resize(const int Width, const int Height); // Plain formats && !Cube
	bool Decrease2X();
	bool MakePowerOfTwo(); // Returns "true" if it is power of two already or has been converted successfully
	bool Make2D(const bool Square = true, const int DepthStep = 1); // Converts 3D texture to 2D set of depth slices
	void CreateSevenLods(const char *SaveAs = "data/textures/SevenLods.dds");
protected:
	int m_Width;
	int m_Height;
	int m_Depth;
	cFormat::Enum m_Format;
	int m_MipMapCount;
	byte *m_Pixels;
};

//-----------------------------------------------------------------------------------
// cImage::GetPixelRgb8
//-----------------------------------------------------------------------------------
inline const cImage::PixelRgb8 cImage::GetPixelRgb8(const int X, const int Y) const {
	cAssert(cFormat::Rgb8 == m_Format);
	
	PixelRgb8 p;
	if(X < 0 || X >= m_Width || Y < 0 || Y >= m_Height) {
		p.r = p.g = p.b = 0;
		return p;
	}
	int Index = 3 * (m_Width * Y + X);
	const byte *Src = &m_Pixels[Index];
	p.r = *Src++;
	p.g = *Src++;
	p.b = *Src++;
	return p;
} // cImage::GetPixelRgb8

//------------------------------------------------------------------------------
// cImage::SetPixelRgb8
//------------------------------------------------------------------------------
inline void cImage::SetPixelRgb8(const int X, const int Y, const PixelRgb8 &p) {
	cAssert(cFormat::Rgb8 == m_Format);

	if(X < 0 || X >= m_Width || Y < 0 || Y >= m_Height) {
		return;
	}
	int Index = 3 * (m_Width * Y + X);
	byte *Src = &m_Pixels[Index];
	*Src = p.r;
	Src++;

	*Src = p.g;
	Src++;

	*Src = p.b;
	Src++;
} // cImage::SetPixelRgb8

//-------------------------------------------------------------------------------------
// cImage::GetPixelRgba8
//-------------------------------------------------------------------------------------
inline const cImage::PixelRgba8 cImage::GetPixelRgba8(const int X, const int Y) const {
	cAssert(cFormat::Rgba8 == m_Format || cFormat::Rgb8 == m_Format);
	PixelRgba8 p;
	if (cFormat::Rgb8 == m_Format){
		int Index = 3 * (m_Width * Y + X);
		const byte *Src = &m_Pixels[Index];
		p.r = *Src++;
		p.g = *Src++;
		p.b = *Src++;
		p.a = 255;
		return p;
	}
	if(X < 0 || X >= m_Width || Y < 0 || Y >= m_Height) {
		p.r = p.g = p.b = p.a = 0;
		return p;
	}
	int Index = 4 * (m_Width * Y + X);
	const byte *Src = &m_Pixels[Index];
	p.r = *Src++;
	p.g = *Src++;
	p.b = *Src++;
	p.a = *Src++;
	return p;
} // cImage::GetPixelRgba8

//--------------------------------------------------------------------------------
// cImage::SetPixelRgba8
//--------------------------------------------------------------------------------
inline void cImage::SetPixelRgba8(const int X, const int Y, const PixelRgba8 &p) {
	cAssert(cFormat::Rgba8 == m_Format);

	if(X < 0 || X >= m_Width || Y < 0 || Y >= m_Height) {
		return;
	}
	int Index = 4 * (m_Width * Y + X);
	byte *Src = &m_Pixels[Index];
	*Src = p.r;
	Src++;

	*Src = p.g;
	Src++;

	*Src = p.b;
	Src++;

	*Src = p.a;
} // cImage::SetPixelRgba8

//-----------------------------------------------------------------------------
// cImage::GetPixelR8
//-----------------------------------------------------------------------------
inline byte cImage::GetPixelR8(const int X, const int Y) const {
	cAssert(cFormat::R8 == m_Format);

	if(X < 0 || X >= m_Width || Y < 0 || Y >= m_Height) {
		return 0;
	}
	int Index = m_Width * Y + X;
	return m_Pixels[Index];
} // cImage::GetPixelR8

// cImage::SetPixelR8
inline void cImage::SetPixelR8(const int X, const int Y, const byte p) {
	cAssert(cFormat::R8 == m_Format);

	if(X < 0 || X >= m_Width || Y < 0 || Y >= m_Height) {
		return;
	}
	int Index = m_Width * Y + X;
	m_Pixels[Index] = p;
}
