// The trees generator example, look the \ref GeneratorExample.cpp for the basic principles
#include <CoreAPI.h>
//@config: Release

using namespace coat;
static const char* backup = "data/Temp/TreesGenerator.json";
const char* GeneratorID = "MyTreesGenerator";


// This is the tree trunk generator class.
// It ir registered as tool in the current room toolset as the first tool in list.
// In voxels additional smoothing spheres are applied to make smooth transition between branches.
// This tool is derived from the SculptGenerator, it means that we make non-destructive object
class TreesGenerator : public SculptGenerator{
public:
	TreesGenerator() {
		defaults();
		collision.setUnit(8);
	}
	// returns the name/ID of the tool in the toolset
	virtual const char* GetID() {
		return GeneratorID;
	}
	void defaults() {
		sign = 1;
		
		noiseAmpliicator = 1.2;
		chunkLengthAmplificator = 1.3;
		overallDensity = 1.0;
		radiusDecrement = 0.75;
		gravity = 0.05;
		transition = 4;
		start_transition = 0;
		ends_distortion = 0.125;
		Level0_branches = 1;

		Level0_Randomness = 1;
		Level0_StartBranchesLevel = 0;
		Level0_Proportions = 1;
		GrowUpStrength = 1;

		BranchesRandomness = 1;
		BranchesStartAngle = 100;
		BranchesEndAngle = 0;
		BranchesProportions = 1;

		MinimalBranchRadius = 0.9;

		MakeRoots = true;
		RootsRandomness = 1;
		RootsLength = 1;
		RootsComplexity = 0.5;
		RootsCount = 8;
		TreeGenSeed = 0;
		RootsStartGrowHeight = 0.05;
	}

	// the summary mesh
	Mesh summ;

	// the spheres to relax the voxels after merging
	list<vec4> relaxSpheres;

	// the collision space to avoid self-intersections
	SphericalCollision collision;

	// the strength of grow-up
	float GrowUpStrength;

	// the points of the main trunk (uniform and dense enough)
	list<vec4> mainTrunk;

	// positive for trunk, negative for roots
	float sign;

	// the noise multiplied by this value when you generate next level of branches
	float noiseAmpliicator;

	// the branches become relatively lengthy when you generate next level
	float chunkLengthAmplificator;

	// overall density, it increases the branches frequency
	float overallDensity;

	// how the rediu decreases when we generate next level
	float radiusDecrement;

	// the anti-gravity level, how much the branches are pulled up
	float gravity;

	// the bigger value means faster transition of the branch direction from the parent direction to the destination direction 
	float transition;

	// the initial transition
	float start_transition;

	// ends of branches are more distorted
	float ends_distortion;

	// randomness of the main branch
	float Level0_Randomness;

	// amount of main branches
	int Level0_branches;

	// the level (0..1) of the first branch
	float Level0_StartBranchesLevel;

	// the main branch proportions 
	float Level0_Proportions;

	// secondary branches curvature
	float BranchesRandomness;

	// initial angle of growing the secondary branches (angle taken between 2 values)
	float BranchesStartAngle;

	// the end value of the angle
	float BranchesEndAngle;

	// seconday branches proportions
	float BranchesProportions;

	// the threshold for the branch radius
	float MinimalBranchRadius;

	// need roots?
	bool MakeRoots;

	// the roots curvature
	float RootsRandomness;

	// the roots proportions (in comparison to default)
	float RootsLength;

	// the sub-branches frequency
	float RootsComplexity;

	// the initial height where roots start growing
	float RootsStartGrowHeight;

	// the count of roots
	int RootsCount;

	// the seed for the random generator
	int TreeGenSeed;

	void RandomizeTree() {
		TreeGenSeed = rand();
		NotifyChanges();
	}

	void SaveTreePreset() {
		str fn;
		if(io::saveFileDialog("*.tree",fn)) {
			WriteToFile(fn);
		}
	}

	void LoadTreePreset() {
		str fn;
		if (io::openFileDialog("*.tree", fn)) {
			defaults();
			ReadFromFile(fn);
		}
		NotifyChanges();
	}

	virtual const char* getDefaultObjectName() {
		return "Tree";
	}

	SERIALIZE() {
		FUNCTION_CALL(CreateNewObject,"CreateNewTree");
		if (getObject()) {
			FSLIDER(overallDensity, "overallDensity", 0, 2, 100, 0);
			FSLIDER(GrowUpStrength, "GrowUpStrength", -5, 5, 100, 0);
			SLIDER(Level0_branches, "Level0_branches", 1, 8);
			FSLIDER(Level0_Randomness, "Level0_Randomness", 0, 3, 100, 0);
			FSLIDER(Level0_StartBranchesLevel, "Level0_StartBranchesLevel", 0, 0.8, 100, 0);
			FSLIDER(Level0_Proportions, "Level0_Proportions", 0, 3, 100, 0);
			DELIMITER;
			FSLIDER(BranchesProportions, "BranchesProportions", 0, 3, 100, 0);
			FSLIDER(radiusDecrement, "radiusDecrement", 0, 0.85, 100, 0);
			FSLIDER(BranchesRandomness, "BranchesRandomness", 0, 3, 100, 0);
			FSLIDER(BranchesStartAngle, "BranchesStartAngle", 30, 160, 1, 0);
			FSLIDER(BranchesEndAngle, "BranchesEndAngle", 30, 160, 1, 0);
			FSLIDER(MinimalBranchRadius, "MinimalBranchRadius", 0.5, 2, 1, 0);
			DELIMITER;
			REG_AUTO(MakeRoots);
			if (MakeRoots) {
				FSLIDER(RootsRandomness, "RootsRandomness", 0, 3, 100, 0);
				FSLIDER(RootsLength, "RootsLength", 0, 3, 100, 0);
				FSLIDER(RootsComplexity, "RootsComplexity", 0, 2, 100, 0);
				FSLIDER(RootsStartGrowHeight, "RootsStartGrowHeight", 0, 0.5, 100, 0);
				SLIDER(RootsCount, "RootsCount", 2, 16);
			}
			DELIMITER;
			UI_LAYOUT("1[]");
			REG_AUTO(TreeGenSeed);
			FUNCTION_CALL(RandomizeTree, "{maticon dice}");
			UI_LAYOUT("2");
			FUNCTION_CALL(SaveTreePreset);
			FUNCTION_CALL(LoadTreePreset);
			FUNCTION_CALL(Generate, "GenerateTreeInGoodQuality");
			REG_AUTO(TransformObject);
			if(TransformObject) {
				NOHASH{
					REG_AUTO(Transform);
				}
			}
		}
	}

	// check the sphere collision
	float checkVoxelsCollision(Volume& v, const vec3& pos, float radius) {
		return collision.collides(pos, radius).LengthSq() > 0;
	}

	// create the branch
	void create(int level, Volume v, vec3 start, vec3 start_direction, vec3 direction, float radius, float rdiff, float relative_length, float randomness) {
		Curve cu;
		vec3 N = start_direction.GetOrthonormal();
		int npoints = 20;
		float L = radius * relative_length / npoints;
		if (sign < 0)L *= RootsLength;
		else if (level == 0)L *= Level0_Proportions;
		else L *= BranchesProportions;
		list<vec3> best;
		float best_cost = FLT_MAX;
		vec3 best_out;
		vec3 dir0 = direction;
		vec3 start0 = start;
		vec3 ort = direction;
		ort.MakeOrthonormalTo(start_direction);
		float random_level = randomness;
		if (sign < 0)random_level *= RootsRandomness;
		else if (level == 0)random_level *= Level0_Randomness;
		else random_level *= BranchesRandomness;
		float this_transition = transition;
		float this_start_transition = start_transition;
		if(level == 0 && sign > 0) {
			this_transition = 0;
			this_start_transition = 1;
		}
		int nattempts = (level == 0 && Level0_branches <= 1) && sign > 0 ? 1 : 30;

		// we are trying many times choosing the random branch that collides with the existing spheres minimally
		for (int k = 0; k < nattempts; k++) {
			direction = dir0;
			// we rotate branch direction randomly, except level 0
			mat4 rot = level > 0 ? mat4::RotationAt(start0, start_direction, comms::cMath::Rand(0, 360)) : mat4::Identity;
			direction.TransformNormal(rot);
			start = start0 + ort * rdiff;
			// we start growing a bit offside, not from the center of the parent
			vec3 out = start0 + ort * (rdiff + radius);
			start.TransformCoordinate(rot);
			out.TransformCoordinate(rot);
			list<vec3> current;
			float cost = 0;
			// the initial growth direction
			vec3 dir = start_direction;
			for (int i = 0; i < npoints; i++) {
				float r = radius * (npoints - i) / npoints;
				if (start.y * sign < 0)cost -= start.y * 1000 * sign;
				current.Add(start);
				cost += checkVoxelsCollision(v, start, r * 1.2);
				// the transition degree between initial parent trunk diretion and the destination direction 
				float t = float(i) * this_transition / npoints + this_start_transition;
				if (t > 1)t = 1;
				dir = start_direction * (1 - t) + direction * t;
				start += L * dir;
				// apply the gravity
				direction.y += gravity;
				direction += vec3::RandNormal() * random_level * (1.0 + float(i) * ends_distortion / 20);
				direction.Normalize();
			}
			if (cost < best_cost) {
				// we choose minimally collided branch
				best_cost = cost;
				best = current;
				best_out = out + start_direction * radius * 2;
			}
		}
		// add points to the curve, we need to canculate curve normal as well
		for (int i = 0; i < best.Count(); i++) {
			float r = radius * (npoints + 1 - i) / (npoints + 1);
			if (r > MinimalBranchRadius) {
				int ip = i - 1;
				int in = i + 1;
				if (ip < 0)ip = 0;
				if (in > best.Count() - 1)in = best.Count() - 1;
				vec3 d = (best[in] - best[ip]).ToNormal();
				N.MakeOrthonormalTo(d);
				cu.add(best[i], N, r);
			}
		}
		// get the render points of the curve (dense)
		cu.updatePoints();
		if (level == 0 && sign > 0) {
			mainTrunk.Clear();
			for (int i = 0; i < cu.renderPointsCount(); i++) {
				auto* a = cu.renderPoint(i);
				mainTrunk.Add(vec4(a->SpacePos, a->Radius));
			}
		}
		Mesh m;
		cu.tubeToMesh(m, true);
		int ns = int(radius * overallDensity / 1.2);
		if (sign < 0)ns = int(radius * RootsComplexity / 1.2);
		randomGrow(level + 1, v, cu, m, best_out, ns, relative_length, randomness);
	}

	// grow next level branches
	void randomGrow(int level, Volume v, Curve& cu, Mesh& m, vec3 outPos, int count, float relative_length, float randomness) {
		list<int> pts;
		//  we gather the points-candidates for growing, they should not be inside the existing branches (excett parent of course), so we check collision
		for (int i = 0; i < count; i++) {
			int nr = cu.renderPointsCount();
			if (nr > 8) {
				int p0 = Level0_StartBranchesLevel * nr;
				if (p0 < 1)p0 = 1;
				if (level > 1 || sign < 0)p0 = 1;

				int s = nr / 16;
				int p = p0 + i * (nr - p0 - 4) / count;
				auto* a = cu.renderPoint(p);
				if (!checkVoxelsCollision(v, a->SpacePos, a->Radius * 1.1)) {
					pts.Add(p);
				}
			}
		}
		// we add collision spheres of the parent branch
		int nr = cu.pointsCount();
		for (int i = 0; i < nr; i++) {
			auto* p = cu.point(i);
			collision.addSphere(p->SpacePos, p->Radius);
		}
		// add the mesh to the resulting mesh
		summ += m;
		if (BranchesEndAngle < BranchesStartAngle)std::swap(BranchesEndAngle, BranchesStartAngle);
		auto* p0 = cu.point(0);
		if (p0 && p0->Radius > 1) {
			float r = p0->Radius;
			if (sign > 0)r *= 2;
			else r *= 1.25;
			relaxSpheres.Add(vec4(outPos, r));			
		}

		for(int i=0;i<pts.Count();i++){
			int p = pts[i];
			auto* a_prev = cu.renderPoint(p - 1);
			auto* a_curr = cu.renderPoint(p);
			auto* a_next = cu.renderPoint(p + 1);
			vec3 dir = (a_next->SpacePos - a_prev->SpacePos).ToNormal();
			vec3 N = a_curr->TempNormal;
			vec3 pos = a_curr->SpacePos;
			mat4 rot = mat4::RotationAt(pos, dir, comms::cMath::Rand(0, 360));
			float ang = float(BranchesStartAngle + comms::cMath::Rand01() * (BranchesEndAngle - BranchesStartAngle)) * comms::cMath::Pi / 180;
			float _cos = cos(ang);
			vec3 tdir = (dir * cos(ang) + N * sin(ang)).ToNormal();
			tdir.TransformNormal(rot);
			create(level + 1, v, pos, dir, tdir
				, a_curr->Radius * radiusDecrement
				, a_curr->Radius * (1.0f - radiusDecrement)
				, relative_length * chunkLengthAmplificator, randomness * noiseAmpliicator);			
		}
	}

	void generate(Volume v, bool Preview) {
		comms::cMath::Randomize(TreeGenSeed);
		summ.geometry()->Clear();
		collision.clear();
		relaxSpheres.Clear();
		mainTrunk.Clear();
		TagsList t;
		Save(t, this);
		gravity = 0.05 * GrowUpStrength;
		for (int i = 0; i < Level0_branches; i++) {
			create(0, v, vec3::Zero, vec3::AxisY, vec3::AxisY, 20, 0, 20, 0.2);
		}
		if (MakeRoots && RootsCount && mainTrunk.Count() > 4) {
			overallDensity *= 0.7;
			sign = -10;
			gravity = 0.01;
			transition = 2;
			start_transition = 0.15;
			ends_distortion = 2.0;

			int a0 = 360 / RootsCount;
			for (int r = 0; r < RootsCount; r++) {
				vec3 ax(1, 0, 0);
				ax *= mat3::Rotation(vec3::AxisY, r * a0 + comms::cMath::Rand(-45, 45));
				ax.Normalize();
				int n0 = mainTrunk.Count() / 30  + 1;
				int g = n0 + int(RootsStartGrowHeight * comms::cMath::Rand(0, mainTrunk.Count() - n0 - 2));
				vec3 d = (mainTrunk[g - 1].ToVec3() - mainTrunk[g + 1].ToVec3()).ToNormal();
				float rad = mainTrunk[g].w;
				create(0, v, mainTrunk[g].ToVec3(), d, ax, rad * 0.6, rad * 0.4, 20, 0.2);
			}
			defaults();
			Load(t, this);
		}
		v.mergeMesh(summ);
	}
	void finish(Volume v) {
		v.mergeMesh(summ);
		for (int k = 0; k < relaxSpheres.Count(); k++) {
			v.relaxGpu(relaxSpheres[k].ToVec3(), relaxSpheres[k].w * 2, 1);
		}
	}	
	virtual void GeneratePreview() {
		SceneElement s(getObject());
		SceneElement t;
		if (s.childCount() == 0) {
			t = s.addChild("Trunk");
			auto v = t.Volume();
			v.assignShader("#Jama Clay/JamaClay1");
			v.setFloatShaderProperty("Textures scale", 1.0);
			v.setFloatShaderProperty("Bumpness", 4.0);
			v.setFloatShaderProperty("Gloss", 0.4);
			v.setFloatShaderProperty("Side rotation [1]", comms::cMath::Rad(0.0f));
			v.setFloatShaderProperty("Side rotation [2]", comms::cMath::Rad(90.0f));
			v.setColorShaderProperty("Color", 0xFF935B38);
			v.setColorShaderProperty("Cavity color", 0xFF2F1D12);
			v.setColorShaderProperty("Bulge color", 0xFF51331D);
			v.setBoolShaderProperty("Flat shading", false);
		}
		else t = s.child(0);
		Volume v = t.Volume();
		v.clearNoUndo();
		v.toSurface();
		t.setTransform(mat4::Identity);
		generate(t.Volume(), true);
		WriteToFile(backup);
		t.setTransform(t.getTransform() * s.getTransform());
		s.selectOne();
	};
	virtual void GenerateFinalObject() {
		SceneElement s(getObject());
		SceneElement t;
		if (s.childCount() == 0)t = s.addChild("Trunk");
		else t = s.child(0);
		Volume v = t.Volume();
		v.clearNoUndo();		
		v.toVoxels();
		t.setTransform(mat4::Identity);
		generate(t.Volume(), true);
		for (int k = 0; k < relaxSpheres.Count(); k++) {
			v.relaxGpu(relaxSpheres[k].ToVec3(), relaxSpheres[k].w * 2, 1);
		}
		t.setTransform(s.getTransform());
		s.selectOne();
	}
};

EXPORT_EXTENSION(TreesGenerator) {
	TreesGenerator* tg = new TreesGenerator;
	tg->ReadFromFile(backup);
	VoxelExtension::Register(tg);
	coat::ui::cmd(str("[extension]") + tg->GetID());
}
