#pragma once

#include "CoreAPI.h"

#define to_scast_obj(T,obj) static_cast<T*>(obj)

#pragma pack(push, 1)

namespace coat {
	using namespace comms;
	typedef One2DCurve curve2d;
	/**
	 * \brief The abstract prim class.
	 */
	class APICALL prim {
	public:
		/**
		 * \brief constructs a new prim object.
		 */
		prim();
		/**
		 * \brief get the primitive class name.
		 */
		str class_name();
		/**
		 * \brief set the primitive object name.
		 * \param s the object name
		 * \return the prim object reference
		 */
		prim& name(const char* s);
		/**
		 * \brief get the primitive object name.
		 */
		str name() const;
		/**
		 * \brief add the prim into scene
		 * \param v the scene volume reference
		 */
		void add(Volume& v);
		/**
		 * \brief subtract the prim from scene
		 * \param v the scene volume reference
		 */
		void subtract(Volume& v);
		/**
		 * \brief intersect the prim into scene
		 * \param v the scene volume reference
		 */
		void intersect(Volume& v);
		/**
		 * \brief merge the prim into scene
		 * \param v the scene volume reference
		 * \param op the type of the merge
		 */
		void merge(Volume& v, const BoolOpType op);
		/**
		 * \brief get the mesh prim
		 * \return mesh object
		 */
		Mesh mesh();
		/**
		 * \brief assign the color to the primitive (in voxels)
		 * \param CL color in hexadecimal form 0xAARRGGBB
		 * \return the reference
		 */
		prim& color(DWORD CL);

		/**
		 * \brief assign the color to the primitive (in voxels)
		 * \param r red value 0..255
		 * \param g green value 0..255
		 * \param b blue value 0..255
		 * \param a alpha value 0..255
		 * \return the reference
		 */
		prim& color(float r, float g, float b, float a);

		/**
		 * \brief assign the color to the primitive (in voxels)
		 * \param r red value 0..255
		 * \param g green value 0..255
		 * \param b blue value 0..255
		 * \return the reference
		 */
		prim& color(float r, float g, float b);

		/**
		 * \brief assign the color to the primitive (in voxels)
		 * \param colorid the color in any suitable form: "RGB", "ARGB", "RRGGBB", "AARRGGBB", "#RGB", "#ARGB", "#RRGGBB", "#AARRGGBB",
		 * any web-color common name as "red", "green", "purple", google "webcolors"
		 * \return the reference
		 */
		prim& color(const char* colorid);

		/**
		 * \brief assign the gloss for the voxel primitive, it will work only if the color already assigned
		 * \param value the [0..1] value of the gloss
		 * \return the reference
		 */
		prim& gloss(float value);

		/**
		 * \brief assign the roughness for the voxel primitive, it will work only if the color already assigned
		 * \param value the [0..1] value of the roughness
		 * \return the reference
		 */
		prim& roughness(float value);

		/**
		 * \brief the metalliclty value for the voxel primitive, it will work only if the color already assigned
		 * \param value the [0..1] metal value
		 * \return the reference
		 */
		prim& metal(float value);

		/**
		 * \brief assign the opacity of the color over the voxel primitive. The color should be assigned before you assign the opacity,
		 * for example p.color("red").opacity(0.5)
		 * \param value the opacity value [0..1]
		 * \return the reference
		 */
		prim& opacity(float value);
		/**
		 * \brief set the detail level
		 * \return prim reference
		 */
		prim& details(const float det_level);
		/**
		 * \brief get the detail level
		 * \return detail level
		 */
		float details();
		/**
		 * \brief set the transform matrix
		 * \param t the matrix
		 * \return prim reference
		 */
		prim& transform(const mat4& t);
		/**
		 * \brief get the transform matrix
		 * \return matrix
		 */
		mat4 transform() const;
		/**
		 * \brief set the scale
		 * \param scale the scale factor
		 * \return prim reference
		 */
		prim& scale(float scale);
		/**
		 * \brief set the scale
		 * \param v the scale vector
		 * \return this primitive reference
		 */
		prim& scale(const vec3& v);
		/**
		 * \brief get the scale
		 * \return the scale 3d vector
		*/
		vec3 scale()  const;
		/**
		 * \brief Set the primitive translation
		 * \param _pos the new primitive position 
		 * \return this primitive reference
		 */
		prim& translate(const vec3& _pos);
		/**
		 * \brief get the primitive translation
		 * \return primitive translation
		 */
		vec3 translate() const;
		/**
		 * \brief Set the primitive translation
		 * \param x the new x primitive position
		 * \param y the new y primitive position
		 * \param z the new z primitive position
		 * \return this primitive reference
		 */
		prim& translate(float x, float y, float z);
		/**
		 * \brief shift the primitive along the x - axis
		 * \param x the x value
		 * \return this primitive reference
		 */
		prim& x(float x);
		/**
		 * \brief shift the primitive along the y - axis
		 * \param y the y value
		 * \return this primitive reference
		 */
		prim& y(float y);
		/**
		 * \brief shift the primitive along the z - axis
		 * \param z the z value
		 * \return this primitive reference
		 */
		prim& z(float z);
		/**
		 * \brief set the auto devide
		 * \param average_div the average divide factor
		 * \return this prim reference
		 */
		prim& auto_divide(float average_div);
		/**
		 * \brief set the step devide
		 * \param step the step divide factor
		 * \return primitive reference
		 */
		prim& step_divide(float step);
		/**
		 * \brief set the fillet 
		 * \param radius the fillet radius
		 * \return this primitive reference
		 */
		prim& fillet(float radius);
		/**
		 * \brief indicates whether to turn on or off the debug mode. 
		 * \param isOn if this parameter is true, the debug mode is on, otherwise the debug mode is off.
		*/
		static void debug_on(bool isOn=true);
		/**
		 * \brief clear the debug info for primitive operations
		*/
		static void debug_clear();
		/**
		 * \brief set the global transform matrix to all primitives
		 * \param t the matrix
		 */
		static void push_transform(const mat4& t);
		/**
		 * \brief Set the translation to all primitives
		 * \param d the new position of the primitives
		 * \details Not implemented yet
		 */
		static void push_translate(const vec3& d);
		/**
		 * \brief Set the scale to all primitives
		 * \param scale the new scale factor
		 * \details Not implemented yet
		 */
		static void push_scale(float scale);
		/**
		 * \brief Set the scale to all primitives
		 * \param scale the new scale 3d vector
		 * \details Not implemented yet
		*/
		static void push_scale(const vec3& s);
		/**
		 * \brief set the detail level to all primitives
		 * \param details_modulator datail level
		 * \details Not implemented yet
		 */
		static void push_details(float details_modulator);
		/**
		 * \brief reset the global transform matrix
		 * \details Not implemented yet
		 */
		static void reset_transform();
		/**
		 * \brief Get the primitive object reference
		 * \return this primitive object reference
		 */
		template <typename T>
		T& Obj() {	return to_cast_this<T>(); }
		/**
		 * \brief Check the prim's type
		 * \return true if the prim's type T is or inherits prim type T.
		 */
		template <typename T>
		bool isAs() { return dynamic_cast<const T*>(this) != NULL; }
		/**
		 * \brief calculates a fillet relative value (0..1).
		 * \return fillet relative value
		 */
		virtual float fillet_relative();
	protected:
		/**
		 * \brief cast this prim to the obj type T 
		 * \return obj type T reference
		 */
		template<class T>
		T& to_cast_this()  { return *static_cast<T*>(this); }
		/**
		 * \brief abstract method to create a mesh container.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh() { return nullptr; }
		/**
		 * \brief merge the mesh debug into scene
		 * \param vol the volume reference
		 * \param mesh the mesh geometry of this prim
		 */
		void merge_debug(Volume& vol, Mesh& mesh);
		prim* _this;
private:
	
		int _nx;
		int _ny;
		int _nz;

		float _average_div;
		float _step_div;

		mat4 _transform;
		float _details;
		str _name;

		static mat4 _ovTransform;
		static float _ovDetails;
		static bool _prmDebug;
		static SceneElement _seDebug;

		DWORD _Color;

		float _fillet;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide() { ; }
		/**
		 * \brief get the fillet relative radius 
		 * \param rmax the maximum radius value
		 * \return radius relative value (0..1)
		 */
		float filletRadiusRelative(float rmax);

		friend class box;
		friend class sphere;
		friend class ellipse;
		friend class cylinder;
		friend class cone;
		friend class tube;
		friend class gear;
		friend class ngon;
		friend class capsule;
		friend class torus;
		friend class spiral;

		template <typename T> T& _src(T* s) {
			auto* t = dynamic_cast<T*>(_this);
			if (t)return *t;
			return *s;
		}
	};
	/**
	 * \brief The box.
	 */
	class APICALL box : public prim {
	public:
		/**
		 * \brief constructs a new box object.
		 */
		box();
		/**
		 * \brief constructs a new box object.
		 */
		/**
		 * \brief constructs a new box object.
		 * \param sizeX size in the x-axis
		 * \param sizeY size in the y-axis 
		 * \param sizeZ size in the z-axis
		 */
		box(float sizeX, float sizeY, float sizeZ);
		/**
		 * \brief constructs a new box object.
		 * \param pos box position
		 * \param size box size
		 */
		box(const vec3& size);
		/**
		 * \brief constructs a new box object.
		 * \param pos box position
		 * \param size box size
		 */
		box(const vec3& pos, const vec3& size);
		/**
		 * \brief constructs a new box object.
		 * \param pos box position
		 * \param size box size
		 * \param fillet round radius value
		 */
		box(const vec3& pos, const vec3& size, float fillet);
		/**
		 * \brief set the x-direction
		 * \param directionX the x-direction
		 * \return box reference
		 */
		box& axis(const vec3& directionX);
		/**
		 * \brief set the x and y direction
		 * \param directionX the x-direction
		 * \param directionY the y-direction
		 * \return box reference
		 */
		box& axis(const vec3& directionX, const vec3& directionY);
		/**
		 * \brief set the x, y and z direction
		 * \param directionX the x-direction
		 * \param directionY the y-direction
		 * \param directionZ the z-direction
		 * \return box reference
		 */
		box& axis(const vec3& directionX, const vec3& directionY, const vec3& directionZ);
		/**
		 * \brief reset the x, y and z direction
		 * \return box reference
		 */
		box& reset_axis();
		/**
		 * \brief get the x-axis
		 * \return vec3 axis
		 */
		vec3 axis_x();
		/**
		 * \brief get the y-axis
		 * \return vec3 axis
		 */
		vec3 axis_y();
		/**
		 * \brief get the z-axis
		 * \return vec3 axis
		 */
		vec3 axis_z();
		/**
		 * \brief set the number deviding
		 * \param nx number deviding along the x-axis
		 * \param ny number deviding along the y-axis
		 * \param nz number deviding along the z-axis
		 * \return box reference
		 */
		box& divide(int nx, int ny, int nz);
		/**
		 * \brief set the box size
		 * \param _size size
		 * \return box reference
		 */
		box& size(const vec3& _size);
		/**
		 * \brief set the box size
		 * \param x size x
		 * \param y size y
		 * \param z size z
		 * \return box reference
		 */
		box& size(float x, float y, float z);
		/**
		 * \brief get the box size.
		 * \return size
		 */
		vec3 size() const;
		/**
		 * \brief calculates a fillet relative value (0..1).
		 * \return fillet relative value
		 */
		virtual float fillet_relative();
	protected:
		/**
		 * \brief initialize the default parameters
		 */
		void init();
		/**
		 * \brief creates a mesh container for the box.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		vec3 _axisx;
		vec3 _axisy;
		vec3 _axisz;
		vec3 _side;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
		friend class torus;
	};
	/**
	 * \brief The torus.
	 */
	class APICALL torus :public box {
	public:
		/**
		 * \brief constructs a new torus object.
		 */
		torus();
		/**
		 * \brief constructs a new torus object.
		 * \param ringRadius ring radius 
		 * \param crossSectionRadius cross section radius.
		 */
		torus(const float& ringRadius, const float& crossSectionRadius);
		/**
		 * \brief constructs a new torus object.
		 * \param pos torus position
		 * \param ringRadius ring radius
		 * \param crossSectionRadius cross section radius.
		 */
		torus(const vec3& pos, const float& ringRadius, const float& crossSectionRadius);
		/**
		 * \brief set the number of slices in the mesh.
		 * \param _slices number of slices.
		 * \return torus reference
		 */
		torus& slices(const int& _slices);
		/**
		 * \brief get the number of slices in the mesh.
		 * \return number of slices
		 */
		int slices() const;
		/**
		 * \brief set the number of rings in the mesh.
		 * \param _rings number of rings.
		 * \return torus reference
		 */
		torus& rings(const int& _rings);
		/**
		 * \brief get the number of rings in the mesh.
		 * \return number of rings
		 */
		int rings() const;
		/**
		 * \brief set the ring radius.
		 * \param r ring radius
		 * \return torus reference
		 */
		torus& radius(const float& r);
		/**
		 * \brief get the ring radius.
		 * \return ring radius
		 */
		float radius()const;
		/**
		 * \brief set the cross section radius.
		 * \param section_r cross section radius
		 * \return torus reference
		 */
		torus& section_radius(const float& section_r);
		/**
		 * \brief get the cross section radius.
		 * \return cross section radius
		 */
		float section_radius() const;
		/**
		 * \brief set the ring diameter.
		 * \param d ring diameter
		 * \return torus reference
		 */
		torus& diameter(const float& d);
		/**
		 * \brief get the ring diameter.
		 * \return ring diameter
		 */
		float diameter()const;
		/**
		 * \brief set the cross section diameter.
		 * \param section_d cross section diameter
		 * \return torus reference
		 */
		torus& section_diameter(const float& section_d);
		/**
		 * \brief get the cross section diameter.
		 * \return cross section diameter
		 */
		float section_diameter() const;
		/**
		* \brief set the flag to create a portion of torus.
		* \param _switch the boolean true/false value
		* \return torus reference
		*/
		torus& sector_on(const bool& _switch);
		/**
		* \brief get the flag of creating a portion of torus. Default = false.
		* \return the sector switch 
		*/
		bool sector_on() const;
		/**
		* \brief When sector is on, specifies the angle for torus slices. Default = 360 degrees.
		* \param angle the angle for torus slices
		* \return torus reference
		*/
		torus& slices_angle(const float& angle);
		/**
		* \brief get the angle for torus slices
		* \return the slices angle
		*/
		float slices_angle() const;
		/**
		* \brief When sector is on, specifies the angle for torus rings. Default = 360 degrees.
		* \param angle the angle for torus rings
		* \return torus reference
		*/
		torus& rings_angle(const float& angle);
		/**
		* \brief get the angle for torus rings
		* \return the rings angle
		*/
		float rings_angle() const;

	protected:
		/**
		 * \brief creates a mesh container for the torus.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		float _ringRadius;
		float _crossSectionRadius;
		float _slicesAngle;
		float _ringsAngle;
		bool _isSector;

		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
	};
	/**
	 * \brief The sphere.
	 */
	class APICALL sphere : public prim {
	public:
	 /**
	 * \brief division modes
	 */
		enum  DivisionMode {
			/// triangle mode
			DIV_TRIANGLE = 0,
			/// meridian mode
			DIV_MERIDIAN = 1,
			/// cube projection mode
			DIV_CUBE = 2
		};
		/**
		 * \brief constructs a new sphere object.
		 */
		sphere();
		/**
		 * \brief constructs a new sphere object with radius.
		 * \param radius sphere radius
		 */
		sphere(float radius);
		/**
		 * \brief constructs a new sphere object with position and radius.
		 * \param pos sphere position
		 * \param radius sphere radius
		 */
		sphere(const vec3& pos, float radius);
		/**
		 * \brief set the radius of the sphere.
		 * \param r radius.
		 * \return sphere reference
		 */
		sphere& radius(const float& r);
		/**
		 * \brief get the radius of the sphere.
		 * \return radius value
		 */
		float radius() const;
		/**
		 * \brief set the diameter of the sphere.
		 * \param d diameter.
		 * \return sphere reference
		 */
		sphere& diameter(const float& d);
		/**
		 * \brief get the diameter of the sphere.
		 * \return diameter value
		 */
		float diameter() const;
		/**
		 * \brief set the degree for subdivision in the mesh.
		 * \param subdiv subdivision.
		 * \return sphere reference
		 */
		sphere& sub_division(const int subdiv);
		/**
		 * \brief get the degree of subdivision triangular or cubic division of the sphere.
		 * \return subdivision degree.
		 */
		int sub_division() const;
		/**
		 * \brief set the division mode for the mesh.
		 * \param divmode mode of the mesh division.
		 * \return sphere reference
		 */
		sphere& sub_div_mode(const sphere::DivisionMode& divmode);
		/**
		 * \brief get the division mode for the mesh.
		 * \return mode of the mesh division
		*/
		sphere::DivisionMode sub_div_mode() const;
		/**
		 * \brief set the number of rings in the mesh.
		 * \param _rings number of rings.
		 * \return sphere reference
		 */
		sphere& rings(const int& _rings);
		/**
		 * \brief get the number of rings in the mesh.
		 * \return number of rings
		 */
		int rings() const;
		/**
		 * \brief set the number of slices in the mesh.
		 * \param _slices number of slices.
		 * \return sphere reference
		 */
		sphere& slices(const int& _slices);
		/**
		 * \brief get the number of slices in the mesh.
		 * \return number of slices
		 */
		int slices() const;
		/**
		* \brief set the flag to create a portion of sphere.
		* \param _switch the boolean true/false value
		* \return sphere reference
		*/
		sphere& sector_on(const bool& _switch);
		/**
		* \brief get the flag of creating a portion of sphere. Default = false.
		* \return the sector switch
		*/
		bool sector_on() const;
		/**
		* \brief When sector is on, specifies the angle where the sphere slice begins.
		* \param angle the angle for sphere slice begin
		* \return sphere reference
		*/
		sphere& slice_from(const float& angle);
		/**
		* \brief get the angle where the sphere slice begins.
		* \return the slice begin angle
		*/
		float slice_from() const;
		/**
		* \brief When sector is on, specifies the angle where the sphere slice ends.
		* \param angle the angle for sphere slice end
		* \return sphere reference
		*/
		sphere& slice_to(const float& angle);
		/**
		* \brief get the angle where the sphere slice ends.
		* \return the slice end angle
		*/
		float slice_to() const;
		/**
		* \brief When sector is on, specifies the angle where the sphere ring begins.
		* \param angle the angle for sphere ring begin
		* \return sphere reference
		*/
		sphere& ring_from(const float& angle);
		/**
		* \brief get the angle where the sphere ring begins.
		* \return the ring begin angle
		*/
		float ring_from() const;
		/**
		* \brief When sector is on, specifies the angle where the sphere ring ends.
		* \param angle the angle for sphere ring end
		* \return sphere reference
		*/
		sphere& ring_to(const float& angle);
		/**
		* \brief get the angle where the sphere ring ends.
		* \return the ring end angle
		*/
		float ring_to() const;
	
	protected:
		/**
		 * \brief creates a mesh container for the sphere.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
		/**
		 * \brief calculates the density-based subdivisions.
		 * \param slices the number of subdivisions around the Z axis
		 * \param rings the number of subdivisions along the Z axis
		 */
		virtual void density_subdiv(int& slices, int& rings);
	private:
		float _radius;
		int _subdiv;
		rect _sector;
		bool _bsector;
		DivisionMode _divmode;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
		friend class ellipse;
	};
	/**
	 * \brief The ellipse.
	 */
	class APICALL ellipse : public sphere {
	public:
		/**
		 * \brief constructs a new ellipse object.
		 */
		ellipse();
		/**
		 * \brief constructs a new ellipse object.
		 * \param pos ellipse position
		 * \param rx radius in the x-direction
		 * \param ry radius in the y-direction
		 * \param rz radius in the z-direction 
		 */
		ellipse(const vec3& pos, float rx, float ry, float rz);
		/**
		 * \brief constructs a new ellipse object.
		 * \param pos ellipse position
		 * \param size ellipse size
		 */
		ellipse(const vec3& pos, const vec3& size);
		/**
		 * \brief constructs a new ellipse object.
		 * \param size ellipse size
		 */
		ellipse(const vec3& size);
		/**
		 * \brief set the axis x, y and z direction
		 * \param directionX the x-direction
		 * \param directionY the y-direction
		 * \param directionZ the z-direction
		 * \return ellipse reference
		 */
		ellipse& axis(const vec3& directionX, const vec3& directionY = vec3::AxisY, const vec3& directionZ = vec3::AxisZ);
		/**
		 * \brief reset the x, y and z directions
		 * \return ellipse reference
		 */
		ellipse& reset_axis();
		/**
		 * \brief set the size of the ellipse.
		 * \param _size size
		 * \return ellipse reference
		 */
		ellipse& size(const vec3& _size);
		/**
		 * \brief get the size of the ellipse.
		 * \return ellipse size
		 */
		vec3 size() const;

	protected:
		/**
		 * \brief initializing of the ellipse.
		 */
		void init();
		/**
		 * \brief creates a mesh container for the ellipse.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
		/**
		 * \brief calculates the density-based subdivisions.
		 * \param slices the number of subdivisions around the Z axis
		 * \param rings the number of subdivisions along the Z axis
		 */
		virtual void density_subdiv(int& slices, int& rings);
	private:
		vec3 _axisx;
		vec3 _axisy;
		vec3 _axisz;
		vec3 _side;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
	};
	/**
	 * \brief The cylinder.
	 */
	class APICALL cylinder : public prim {
	public:
		/**
		 * \brief constructs a new cylinder object.
		 */
		cylinder();
		/**
		 * \brief constructs a new cylinder object.
		 * \param height cylinder height
		 * \param radiusTop radius at top
		 * \param radiusBottom radius at bottom
		 * \param fillet fillet radius
		 */
		cylinder(const float& height, const float& radiusTop, const float& radiusBottom, const float& fillet=0.0);
		/**
		* \brief constructs a new cylinder object.
		* \param posTop position at top
		* \param posBottom position at bottom
		* \param radiusTop radius at top
		* \param radiusBottom radius at bottom
		* \param fillet fillet radius
		*/
		cylinder(const vec3& posTop, const vec3& posBottom, const float& radiusTop, const float& radiusBottom, const float& fillet=0.0);
		/**
		* \brief set the top position.
		* \param pos position
		* \return cylinder reference
		*/
		cylinder& positionTop(const vec3& pos);
		/**
		* \brief get the top position.
		* \return position
		*/
		vec3 positionTop() const;
		/**
		* \brief set the bottom position.
		* \param pos position
		* \return cylinder reference
		*/
		cylinder& positionBottom(const vec3& pos);
		/**
		* \brief get the bottom position.
		* \return position
		*/
		vec3 positionBottom() const;
		/**
		* \brief set the top radius.
		* \param r radius
		* \return cylinder reference
		*/
		cylinder& radiusTop(const float& r);
		/**
		* \brief get the top radius.
		* \return radius value
		*/
		float radiusTop() const;
		/**
		* \brief set the bottom radius.
		* \param r radius
		* \return cylinder reference
		*/
		cylinder& radiusBottom(const float& r);
		/**
		* \brief get the bottom radius.
		* \return radius value
		*/
		float radiusBottom() const;
		/**
		* \brief set the radius.
		* \param r radius
		* \return cylinder reference
		*/
		cylinder& radius(const float& r);
		/**
		* \brief get the radius.
		* \return radius value
		*/
		float radius() const;
		/**
		* \brief set the top diameter.
		* \param d diameter
		* \return cylinder reference
		*/
		cylinder& diameterTop(const float& d);
		/**
		* \brief get the top diameter.
		* \return diameter value
		*/
		float diameterTop() const;
		/**
		* \brief set the bottom diameter.
		* \param d diameter
		* \return cylinder reference
		*/
		cylinder& diameterBottom(const float& r);
		/**
		* \brief get the bottom diameter.
		* \return diameter value
		*/
		float diameterBottom() const;
		/**
		* \brief set the diameter.
		* \param d diameter
		* \return cylinder reference
		*/
		cylinder& diameter(const float& d);
		/**
		* \brief get the diameter.
		* \return diameter value
		*/
		float diameter() const;
		/**
		* \brief set the height in the z-axis.
		* \param _height height
		* \return cylinder reference
		*/
		cylinder& height(const float& _height);
		/**
		* \brief get the height.
		* \return height value
		*/
		float height() const;
		/**
		* \brief set the angle for cylinder with sector.
		* \param angle the sector angle
		* \return cylinder reference
		*/
		cylinder& sectorAngle(const float& angle);
		/**
		* \brief get the sector angle.
		* \return angle value
		*/
		float sectorAngle() const;
		/**
		* \brief set the top cap scale.
		* \param scale the scale value
		* \return cylinder reference
		*/
		cylinder& topCapScale(const float& scale);
		/**
		 * \brief get the top cap scale.
		 * \return the scale value
		 */
		float topCapScale() const;
		/**
		* \brief set the bottom cap scale.
		* \param scale the scale value
		* \return cylinder reference
		*/
		cylinder& bottomCapScale(const float& scale);
		/**
		 * \brief get the bottom cap scale.
		 * \return the scale value
		 */
		float bottomCapScale() const;
		/**
		 * \brief set the number of slices in the mesh.
		 * \param nslices number of slices.
		 * \return cylinder reference
		 */
		cylinder& slices(const int& nslices);
		/**
		 * \brief get the number of slices in the mesh.
		 * \return number of slices.
		 */
		int slices() const;
		/**
		 * \brief set the number of rings in the mesh.
		 * \param nrings number of rings.
		 * \return cylinder reference
		 */
		cylinder& rings(const int& nrings);
		/**
		 * \brief get the number of rings in the mesh.
		 * \return number of rings.
		 */
		int rings() const;
		/**
		 * \brief set the number of caps in the mesh.
		 * \param ncaps number of caps.
		 * \return cylinder reference
		 */
		cylinder& caps(const int& ncaps);
		/**
		 * \brief get the number of caps in the mesh.
		 * \return number of caps.
		 */
		int caps() const;
		/**
		 * \brief calculates a fillet relative value (0..1).
		 * \return fillet relative value
		 */
		virtual float fillet_relative();
	protected:
		/**
		 * \brief creates a mesh container for the cylinder.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
		/**
		 * \brief get a mesh container for the cylinder without account of fillet.
		 * \param iscone cone flag 
		 * \return the new mesh container
		 */
		cMeshContainer* getCylinderPrim(bool iscone = false);
		/**
		 * \brief get a transform matrix between standard cylinder and cylinder with custom parameters
		 * \return transform matrix
		 */
		mat4 getLTransform();
	private:
		vec3  _positionTop;
		vec3  _positionBottom;
		float  _radiusTop;
		float  _radiusBottom;
		float _sectorAngle;
		float _topCapScale;
		float _bottomCapScale;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
		friend class cone;
		friend class capsule;
		friend class ngon;
		friend class tube;
	};
	/**
	 * \brief The cone.
	 */
	class APICALL cone : public cylinder {
	public:
		/**
		 * \brief constructs a new cone object.
		 */
		cone();
		/**
		 * \brief constructs a new cone object.
		 * \param height cone height
		 * \param radiusBottom bottom radius
		 * \param fillet fillet radius
		 */
		cone(const float& height, const float& radiusBottom, const float& fillet = 0.0);
		/**
		* \brief constructs a new cone object.
		* \param posTop top position
		* \param posBottom bottom position
		* \param radiusBottom bottom radius
		* \param fillet fillet radius
		*/
		cone(const vec3& posTop, const vec3& posBottom, const float& radiusBottom, const float& fillet = 0.0);
		/**
		* \brief set the value of radius.
		* \param r radius
		* \return cone reference
		*/
		cone& radius(const float& r);
		/**
		* \brief get the value of radius.
		* \return radius value
		*/
		float radius() const;
		/**
		* \brief set the value of diameter.
		* \param d diameter
		* \return cone reference
		*/
		cone& diameter(const float& d);
		/**
		* \brief get the value of diameter.
		* \return diameter value
		*/
		float diameter() const;
		/**
		 * \brief calculates a fillet relative value (0..1).
		 * \return fillet relative value
		 */
		float fillet_relative() override;
	protected:
		/**
		 * \brief creates a mesh container for the cone.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
	};
	/**
	 * \brief The tube.
	 */
	class APICALL tube : public cylinder {
	public:
		/**
		 * \brief constructs a new tube object.
		 */
		tube();
		/**
		 * \brief constructs a new tube object.
		 * \param height cone height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 */
		tube(const float& height, const float& radiusTop, const float& radiusBottom);
		/**
		 * \brief constructs a new tube object.
		 * \param height tube height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 * \param fillet fillet radius
		 */
		tube(const float& height, const float& radiusTop, const float& radiusBottom, const float& fillet);
		/**
		 * \brief constructs a new tube object.
		 * \param height cone height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 * \param relativeHoleRadius relative hole radius (0..1)
		 * \param fillet fillet radius
		 */
		tube(const float& height, const float& radiusTop, const float& radiusBottom, const float& relativeHoleRadius, const float& fillet);
		/**
		* \brief constructs a new tube object.
		* \param posTop top position
		* \param posBottom bottom position
		* \param radiusTop top radius
		* \param radiusBottom bottom radius
		* \param relativeHoleRadius relative hole radius (0..1)
		*/
		tube(const vec3& posTop, const vec3& posBottom, const float& radiusTop, const float& radiusBottom, const float& relativeHoleRadius);
		/**
		* \brief constructs a new tube object.
		* \param posTop top position
		* \param posBottom bottom position
		* \param radiusTop top radius
		* \param radiusBottom bottom radius
		* \param relativeHoleRadius relative hole radius (0..1)
		* \param fillet fillet radius
		*/
		tube(const vec3& posTop, const vec3& posBottom, const float& radiusTop, const float& radiusBottom, const float& relativeHoleRadius, const float& fillet);
		/**
		* \brief set the relative value of the hole radius.
		* \param r relative value
		* \return tube reference
		*/
		tube& relativeHoleRadius(const float& r);
		/**
		* \brief get the relative value of the hole radius.
		* \return relative value (0..1)
		*/
		float relativeHoleRadius() const;
		/**
		* \brief set the wall thickness value.
		* \param w thickness value
		* \return tube reference
		*/
		tube& thickness(const float& w);
		/**
		* \brief get the relative value of the hole radius.
		* \return relative value (0..1)
		*/
		float thickness() const;
		/**
		 * \brief calculates a fillet relative value (0..1).
		 * \return fillet relative value
		 */
		float fillet_relative() override;
	protected:
		/**
		 * \brief creates a mesh container for the tube.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		float _relativeHoleRadius;
		float _thickness;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
	};
	/**
	 * \brief The gear.
	 */
	class APICALL gear : public tube {
	public:
		/**
		 * \brief constructs a new gear object.
		 */
		gear();
		/**
		 * \brief constructs a new gear object.
		 * \param height cone height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 */
		gear(const float& height, const float& radiusTop, const float& radiusBottom);
		/**
		 * \brief constructs a new gear object.
		 * \param height cone height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 * \param depth depth value
		 * \param sharpness sharpness value 
		 * \param order the number of teeth
		 */
		gear(const float& height, const float& radiusTop, const float& radiusBottom, float depth, float sharpness = 0.5, const int& order = 16);
		/**
		* \brief constructs a new gear object.
		* \param posTop top position
		* \param posBottom bottom position
		* \param radiusTop top radius
		* \param depth depth value
		* \param sharpness sharpness value 
		* \param order the number of teeth
		*/
		gear(const vec3& posTop, const vec3& posBottom, const float& radiusTop, const float& radiusBottom, float depth = 0.1, float sharpness = 0.5, const int& order = 16);
		/**
		 * \brief set the depth value.
		 * \param __depth depth value
		 * \return gear reference
		 */
		gear& depth(const float& __depth);
		/**
		* \brief get the depth value.
		* \return depth value
		*/
		float depth() const;
		/**
		 * \brief set the sharpness value.
		 * \param sharp sharpness value
		 * \return gear reference
		 */
		gear& sharpness(const float& sharp);
		/**
		* \brief get the depth value.
		* \return depth value
		*/
		float sharpness() const;
		/**
		 * \brief set the number of teeth in gear.
		 * \param nteeth number of teeth
		 * \return gear reference
		 */
		gear& order(const int& nteeth);
		/**
		* \brief get the number of teeth in gear.
		* \return the number of teeth
		*/
		int order() const;
	protected:
		/**
		 * \brief creates a mesh container for the gear.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		float  _depth;
		float  _sharpness;
		int    _order;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
		friend class ngon;
	};
	/**
	 * \brief The ngon.
	 */
	class APICALL ngon : public gear {
	public:
		/**
		 * \brief constructs a new ngon object.
		 */
		ngon();
		/**
		 * \brief constructs a new ngon object.
		 * \param height ngon height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 * \param order the number of sides
		 */
		ngon(const float& height, const float& radiusTop, const float& radiusBottom, const int& order);
		/**
		 * \brief constructs a new ngon object.
		 * \param height ngon height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 * \param fillet fillet radius
		 * \param order the number of sides
		 */
		ngon(const float& height, const float& radiusTop, const float& radiusBottom, const float& fillet, const int& order);
		/**
		 * \brief constructs a new gear object.
		 * \param posTop top position
		 * \param posBottom bottom position
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 * \param order the number of teeth
		 */
		ngon(const vec3& posTop, const vec3& posBottom, const float& radiusTop, const float& radiusBottom, const int& order);
		/**
		 * \brief calculates a fillet relative value (0..1).
		 * \return fillet relative value
		 */
		virtual float fillet_relative();
	protected:
		/**
		 * \brief creates a mesh container for the ngon.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	};
	/**
	 * \brief The capsule.
	 */
	class APICALL capsule :public cylinder {
	public:
		/**
		 * \brief constructs a new capsule object.
		 */
		capsule();
		/**
		 * \brief constructs a new capsule object.
		 * \param height capsule height
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 */
		capsule(const float& height, const float& radiusTop, const float& radiusBottom);
		/**
		 * \brief constructs a new capsule object.
		 * \param posTop top position
		 * \param posBottom bottom position
		 * \param radiusTop top radius
		 * \param radiusBottom bottom radius
		 */
		capsule(const vec3& posTop, const vec3& posBottom, const float& radiusTop, const float& radiusBottom);
	protected:
		/**
		 * \brief creates a mesh container for the capsule.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
	};

	/**
		* \brief The profile type of the spiral.
		*/
	enum class SpiralProfile {
		/// circle
		CIRCLE = 0,
		/// rectangle
		RECTANGLE = 1
	};

	/**
	 * \brief The spiral.
	 */
	class APICALL spiral:public prim {
	public:
		/**
		 * \brief constructs a new spiral object.
		 */
		spiral();
		/**
		 * \brief constructs a new spiral object.
		 * \param out_radius outer radius
		 * \param in_radius profle radius
		 * \param _sstep spiral step
		 * \param nturns the number of turns
		 */
		spiral(const float& out_radius, const float& in_radius, const float& _sstep, const float& nturns);
		/**
		 * \brief set the outer radius of the spiral.
		 * \param r radius value
		 * \return spiral reference
		 */
		spiral& radius(const float& r);
		/**
		* \brief get the outer radius.
		* \return outer radius
		*/
		float radius() const;
		/**
		 * \brief set the profile radius.
		 * \param r radius value
		 * \return spiral reference
		 */
		spiral& profile_radius(const float& r);
		/**
		* \brief get the profile radius.
		* \return profile radius
		*/
		float profile_radius() const;
		/**
		* \brief set the outer diameter of the spiral.
		* \param d diameter value
		* \return spiral reference
		*/
		spiral& diameter(const float& d);
		/**
		* \brief get the outer diameter.
		* \return outer diameter
		*/
		float diameter() const;
		/**
		 * \brief set the profile diameter.
		 * \param d diameter value
		 * \return spiral reference
		 */
		spiral& profile_diameter(const float& d);
		/**
		* \brief get the profile diameter.
		* \return profile diameter
		*/
		float profile_diameter() const;
		/**
		 * \brief set the number of turns.
		 * \param nturns turns count
		 * \return spiral reference
		 */
		spiral& turns(const int nturns);
		/**
		* \brief get the number of turns.
		* \return turns count
		*/
		int turns() const;
		/**
		 * \brief set the spiral step.
		 * \param vstep step value
		 * \return spiral reference
		 */
		spiral& step(const float& vstep);
		/**
		* \brief get the spiral step.
		* \return step value
		*/
		float step() const;
		/**
		 * \brief set the type of profile (circle or rectangle).
		 * \param type profile type
		 * \return spiral reference
		 */
		spiral& profile_type(const SpiralProfile type);
		/**
		 * \brief set the dimensions for the rectangle profile.
		 * \param width the width value
		 * \param height the height value
		 * \return spiral reference
		 */
		spiral& profile_rect(const float width, const float height);
		/**
		 * \brief set the clockwise direction of the spiral.
		 * \param clockWise the indicator of the clockwise direction
		 * \return spiral reference
		 */
		spiral& clock_wise(const bool& clockWise);
		/**
		* \brief get the clokwise direction of the spiral.
		* \return the clokwise direction( true or false)
		*/
		bool clock_wise() const;
		/**
		* \brief get the profile height for rectangle profile.
		* \return the profile height
		*/
		float profile_height() const;
		/**
		* \brief get the profile width for rectangle profile.
		* \return the profile width
		*/
		float profile_width() const;
		/**
		 * \brief set the number of slices in the mesh.
		 * \param nslices number of slices.
		 * \return spiral reference
		 */
		spiral& slices(const int& nslices);
		/**
		 * \brief get the number of slices in the mesh.
		 * \return number of slices
		 */
		int slices() const;
		/**
		 * \brief set the number of rings in the mesh.
		 * \param nrings number of rings.
		 * \return spiral reference
		 */
		spiral& rings(const int& nrings);
		/**
		 * \brief get the number of rings in the mesh.
		 * \return number of rings
		 */
		int rings() const;
		/**
		 * \brief set the number of caps in the mesh.
		 * \param ncaps number of caps.
		 * \return spiral reference
		 */
		spiral& caps(const int& ncaps);
		/**
		 * \brief get the number of caps in the mesh.
		 * \return number of caps
		 */
		int caps() const;
	protected:
		/**
		 * \brief creates a mesh container for the spiral.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		int _turns;
		float _step;
		float _radius;
		float _profile_radius;
		float _profile_width;
		float _profile_height;
		bool _clock_wise;
		SpiralProfile _profile_type;
		/**
		 * \brief calculates the average deviding.
		 */
		virtual void average_devide();
	};

	/**
		 * \brief Enumeration of the string drawing styles.
		 */
	enum FontStyle {
		Regular = 0,
		Italic = 1 << 0,
		Underline = 1 << 1,
		StrikeThrough = 1 << 2
	};

	enum FontWeight {
		Dont = 0,
		Thin = 100,
		ExtraLight = 200,
		Light = 300,
		Normal = 400,
		Medium = 500,
		DemiBold = 600,
		Bold = 700,
		ExtraBold = 800,
		Black = 900
	};

	/**
		 * \brief Holds the general information about font
		 */
	class APICALL FontInfo {
	public:
		int			size;
		int			weight;
		int			style;
		std::string	fname;
	};

	class APICALL Font {
	public:

		/**
		 * \brief The Font class specifies a query for a font used for drawing text.
		 */
		Font();
		/**
		 * \brief constructs a new font object.
		 * \param _finfo font information 
		 */
		Font(const FontInfo& _finfo);
		/**
		 * \brief constructs a new font object.
		 * \param _fname font name
		 */
		Font(const char* _fname);
		/**
		 * \brief constructs a new font object.
		 * \param _fname the font name
		 * \param _st the font style
		 */
		Font(const char* _fname, const int& _st);
		/**
		 * \brief set the font size
		 * \param _size font size property
		 * \return font reference
		 */
		Font& size(const int& _size);
		/**
		 * \brief get the font size
		 * \return font size
		 */
		int size() const;
		/**
		 * \brief set the font weight
		 * \param weight font weight property
		 * \return font reference
		 */
		Font& weight(const int& weight);
		/**
		 * \brief get the font weight
		 * \return font weight
		 */
		int weight() const;
		/**
		 * \brief set the style of the font
		 * \param st the new font style
		 * \return font reference
		 */
		Font& style(const int st);
		/**
		 * \brief get the font style
		 * \return the font style
		 */
		int style() const;
		/**
		 * \brief set the font name
		 * \param _fname the font name
		 * \return font reference
		 */
		Font& name(const char* _fname);
		/**
		 * \brief get the font name
		 * \return the font name 
		 */
		const char* name() const;
		/**
		 * \brief selects a font object into the viewport
		 */
		void select();
	private:
		/** 
		 * \brief initializing of the font info
		 */
		void init();
		/**
		 * \brief save the general information about font
		 */
		void save(void);
		/**
		* \brief gets the information about font.
		* \return font information
		*/
		FontInfo& info() { return _finfo; }
		const FontInfo& info() const { return _finfo; }
		FontInfo _finfo;
	};

	/**
	* \brief text primitive
	*/
	class APICALL text :public prim {
	public:
		/**
		* \brief constructs a new text object.
		*/
		text();
		/**
		* \brief constructs a new text object.
		* \param s the input text's string
		*/
		text(const char* s);
		/**
		* \brief set the text's string.
		* \param s the string
		*/
		text& string(const char* s);
		/**
		* \brief get the text's string.
		* \return the string
		*/
		str string() const;
		/**
		* \brief set the text font
		* \param f font object
		* \return the text reference
		*/
		text& font(const Font& f);
		/**
		* \brief get the font object
		* \return font object
		*/
		Font font() const;
		/**
		* \brief set the text width in the pixels
		* \param w width
		* \return the text reference
		*/
		text& width(const float& w);
		/**
		* \brief get the text width
		* \return the width value
		*/
		float width() const;
		/**
		* \brief set the text depth in the pixels
		* \param d depth
		* \return the text reference
		*/
		text& depth(const float& d);
		/**
		* \brief get the text depth
		* \return the depth value
		*/
		float depth() const;
		/**
		 * \brief set the bend radius.
		 * \param radius bend radius of the text
		 * \return the text reference
		*/
		text& bendRadius(const float& radius);
		/**
		 * \brief get the bend radius.
		 * \return the bend radius of the text
		*/
		float bendRadius() const;
		/**
		 * \brief set the rotate angle around the x-axis.
		 * \param radius bend radius of the text
		 * \return the text reference
		 */
		text& extraRotation(const float& rotation);
		/**
		 * \brief get the rotate angle around the x-axis.
		 * \return the rotate angle
		*/
		float extraRotation() const;
		/**
		* \brief set the invert of the text bending.
		* \param binvert boolean value true/false
		* \return the text reference
		*/
		text& invertBending(const bool& _binvert);
		/**
		 * \brief get the invert of the text bending.
		 * \return the invert bending
		*/
		float invertBending() const;
	protected:
		/**
		 * \brief creates a mesh container for the text.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
		/**
		 * \brief update the mesh container for the text.
		 * \param mc mesh pointer
		 */
		void update_mesh(cMeshContainer* mc);
	private:
		str _string;
		Font _font;
		vec2 _size;
		float _rotation;
		float _bendRadius;
		bool _invertBending;
		/**
		* \brief initializing of the text
		*/
		void init();
	};
	/**
	* \brief lathe primitive
	*/
	class APICALL lathe :public box {
		bool untouched;
	public:
		/**
		* \brief enumeration of the point state for a lathe profile.
		*/
		enum pointState {
			regular=0,
			bspline=1,
			sharp=2,
			locked=3,
			vertical=4,
			horizontal=5
		};
		/**
		* \brief enumeration of the lathe type.
		*/
		enum Type {
			cylinder=0,
			rectangle=1
		};
		/**
		* \brief constructs a new lathe object.
		*/
		lathe();
		/**
		* \brief object destructor.
		*/
		~lathe();
		/**
		* \brief set the lathe type
		* \param t type value (cylinder or rectangle)
		* \return the lathe reference
		*/
		lathe& type(const lathe::Type& t);
		/**
		 * \brief get the lathe type.
		 * \return the type value
		*/
		lathe::Type type() const;
		/**
		* \brief add the point into curve
		* \param point the 2d point
		* \param st the point state
		* \return lathe reference
		*/
		lathe& add_point(const vec2& point, int st);
		/**
		* \brief get the pointer to the profile
		* \return the profile pointer 
		*/
		curve2d* profile() const;
		/**
		* \brief reset the curve points
		*/
		lathe& reset();
		/**
		* \brief clear points of the profile
		*/
		lathe& clear();
	protected:
		/**
		 * \brief creates a mesh container for the text.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		Type _type;
		curve2d* _profile;
		/**
		 * \brief initializing of the lathe info
		 */
		void init();
	};

	/**
	* \brief image primitive
	*/
	class APICALL image :public text {
	public:
		/**
		 * \brief Holds the general information about image parameters.
		 */
		struct ParamInfo {
			str topTexture;
			str topBumpTexture;
			str bottomTexture;
			str bottomBumpTexture;
			str strencilTexture;
			str bottomStrencilTexture;
			float basicThickness;
			float bumpThickness;
			float taperAngle;
			float topBottomWeight;
			float sizeInScene;
		};
		/**
		* \brief constructs a new image object.
		*/
		image();
		/**
		* \brief set the top texture 
		* \param _texture image file name 
		* \return the image reference
		*/
		image& topTexture(const char* _texture);
		/**
		* \brief get the top texture
		* \return the string of the image file name
		*/
		str topTexture() const;
		/**
		* \brief set the top bump texture
		* \param _texture image file name
		* \return the image reference
		*/
		image& topBumpTexture(const char* _texture);
		/**
		* \brief get the top bump texture
		* \return the string of the image file name
		*/
		str topBumpTexture() const;
		/**
		* \brief set the bottom texture
		* \param _texture image file name
		* \return the image reference
		*/
		image& bottomTexture(const char* _texture);
		/**
		* \brief get the bottom texture
		* \return the string of the image file name
		*/
		str bottomTexture() const;
		/**
		* \brief set the bottom bump texture
		* \param _texture image file name
		* \return the image reference
		*/
		image& bottomBumpTexture(const char* _texture);
		/**
		* \brief get the bottom bump texture
		* \return the string of the image file name
		*/
		str bottomBumpTexture() const;
		/**
		* \brief set the strencil texture
		* \param _texture image file name
		* \return the image reference
		*/
		image& strencilTexture(const char* _texture);
		/**
		* \brief get the strencil texture
		* \return the string of the image file name
		*/
		str strencilTexture() const;
		/**
		* \brief set the bottom strencil texture
		* \param _texture image file name
		* \return the image reference
		*/
		image& bottomStrencilTexture(const char* _texture);
		/**
		* \brief get the bottom strencil texture
		* \return the string of the image file name
		*/
		str bottomStrencilTexture() const;
		/**
		* \brief set the basic thickness of image
		* \param _thickness thickness value 
		* \return the image reference
		*/
		image& basicThickness(const float& _thickness);
		/**
		* \brief get the basic thickness of image
		* \return the thickness value
		*/
		float basicThickness() const;
		/**
		* \brief set the bump thickness of image
		* \param _thickness thickness value
		* \return the image reference
		*/
		image& bumpThickness(const float& _thickness);
		/**
		* \brief get the bump thickness of image
		* \return the thickness value
		*/
		float bumpThickness() const;
		/**
		* \brief set the angle of tapering 
		* \param _angle taper angle value
		* \return the image reference
		*/
		image& taperAngle(const float& _angle);
		/**
		* \brief get the angle of tapering of image
		* \return the taper angle value
		*/
		float taperAngle() const;
		/**
		* \brief set the weight of the top and bottom image
		* \param weight weight value
		* \return the image reference
		*/
		image& topBottomWeight(const float& weight);
		/**
		* \brief get the weight of the top and bottom image
		* \return the weight value
		*/
		float topBottomWeight() const;
		/**
		* \brief set the size of image in the scene
		* \param _size size value
		* \return the image reference
		*/
		image& sizeInScene(const float& _size);
		/**
		* \brief get the size of image in the scene
		* \return the size value
		*/
		float sizeInScene() const;

	protected:
		/**
		 * \brief creates a mesh container for the image.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		ParamInfo _paramInfo;
		/**
		* \brief gets the param information.
		* \return font information
		*/
		ParamInfo& paramInfo() { return _paramInfo; }
		const ParamInfo& paramInfo() const { return _paramInfo; }
	};

	/**
		* \brief thread profile types
		*/
	enum ThreadProfile {
		ThreadNone = -1,
		ThreadTriangle = 0,
		ThreadTrapeze = 1,
		ThreadRectangular = 2,
		ThreadRound = 3,
		ThreadPersistent = 4
	};

	/**
	* \brief thread primitive
	*/
	class APICALL thread :public prim {
	public:
		/**
		 * \brief constructor
		 */
		thread();
		/**
		 * \brief set the diameter of the thread.
		 * \param d diameter value
		 * \return thread reference
		 */
		thread& diameter(const float d);
		/**
		 * \brief get the diameter of the thread.
		 * \return thread diameter
		 */
		float diameter() const;
		/**
		 * \brief set the pitch of the thread.
		 * \param p pitch value
		 * \return thread reference
		 */
		thread& pitch(const float p);
		/**
	     * \brief set the pitch of the thread.
		 * \return pitch value
		 */
		float pitch() const;
		/**
		 * \brief set the stub length of the thread.
		 * \param l stub length value
		 * \return thread reference
		 */
		thread& stub(const float& l);
		/**
		 * \brief get the stub length of the thread.
		 * \return stub length value
		 */
		float stub() const;
		/**
		 * \brief set the height of the thread.
		 * \param h height value
		 * \return thread reference
		 */
		thread& height(const float& h);
		/**
		 * \brief get the height of the thread.
		 * \return height value
		 */
		float height() const;
		/**
		 * \brief set the number of the thread turns.
		 * \param n turns count
		 * \return thread reference
		 */
		thread& turns(const int& n);
		/**
		 * \brief get the number of the thread turns.
		 * \return turns count
		 */
		int turns() const;
		/**
		 * \brief set the clockwise flag of the thread.
		 * \param cw if value is true then thread direction is clockwise
		 * \return thread reference
		 */
		thread& clockwise(const bool& cw);
		/**
		 * \brief get the clockwise of the thread.
		 * \return clockwise flag
		 */
		bool clockwise() const;
		/**
		 * \brief set the closed thread.
		 * \param b closed value
		 * \return thread reference
		 */
		thread& close(const bool& b);
		/**
		 * \brief set the closed thread.
		 * \return closed flag
		 */
		bool close() const;
		/**
		 * \brief set the thread profile type.
		 * \param prf profile type
		 * \return thread reference
		 */
		thread& profile(const ThreadProfile& prf);
		/**
		 * \brief get the thread profile type.
		 * \return profile type
		 */
		ThreadProfile profile() const;

	protected:
		/**
		 * \brief creates a mesh container for the thread.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		ThreadProfile _profile;
		float _diameter;
		float _pitch;
		float _stub;
		float _height;
		bool _close;
		bool _clockwise;
		int _turns;
		friend class threadStud;
	};

	/**
		* \brief thread body type
		*/
	enum ThreadStudBodyType {
		StudCylinder = 0,
		StudCone = 1
	};

	/**
	* \brief thread stud primitive
	*/
	class APICALL threadStud : public thread {
	public:

		/**
		* \brief constructor
		*/
		threadStud();
		/**
		 * \brief set the diameter of the thread.
		 * \param d diameter value
		 * \return thread stud reference
		 */
		threadStud& diameter(const float& d);
		/**
		 * \brief get the diameter of the thread.
		 * \return diameter value
		 */
		float diameter() const;
		/**
		 * \brief get the top diameter of the thread.
		 * \param d diameter value
		 * \return thread stud reference
		 */
		threadStud& diameterTop(const float& d);
		/**
		 * \brief get the top diameter of the thread.
		 * \return diameter value
		 */
		float diameterTop() const;
		/**
		 * \brief set the bottom diameter of the thread.
		 * \param d diameter value
		 * \return thread stud reference
		 */
		threadStud& diameterBottom(const float& d);
		/**
		 * \brief get the bottom diameter of the thread.
		 * \return diameter value
		 */
		float diameterBottom() const;
		/**
		 * \brief set the stud length.
		 * \param l length value
		 * \return thread stud reference
		 */
		threadStud& length(const float& l);
		/**
		 * \brief get the length of the stud.
		 * \return length value
		 */
		float length() const;
		/**
		 * \brief get the length of the thread.
		 * \return length value
		 */
		float threadLength() const;
		/**
		 * \brief enabled or disabled thread in the stud.
		 * \param bEnable Indicates whether to enable or disable the thread.\ 
		 * If this parameter is true, the thread is enabled. If the parameter is false, the thread is disabled.
		 * \return thread stud reference
		 */
		threadStud& enableThread(const bool& enable);
		/**
		 * \brief get the indicator of the enabled thread.
		 * \return enabled/disabled value
		 */
		bool enableThread() const;
		/**
		 * \brief set the body type.
		 * \param bt body type
		 * \return thread stud reference
		 */
		threadStud& bodyType(const ThreadStudBodyType& bt);
		/**
		 * \brief get the body type.
		 * \return type value
		 */
		ThreadStudBodyType bodyType() const;

	protected:
		/**
		 * \brief creates a mesh container for the thread stud.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		float _diameterTop;
		float _diameterBottom;
		float _length;
		bool _enableThread;
		ThreadStudBodyType _bodyType;
	};

	/**
		 * \brief enumeration the slit types
		 */
	enum SlitType {
		none = -1,
		Slot = 0,
		Phillipse = 1,
		Pozidriv = 2,
		Robertson = 3,
		HexSocket = 4,
		SecurityHexSocket = 5,
		Torx = 6,
		SecurityTorx = 7,
		TriWing = 8,
		TorqSet = 9,
		TripleSquare = 10,
		Polydrive = 11,
		DoubleSquare = 12,
		SplineDrive = 13,
		DoubleHex = 14,
		Bristol = 15,
		Pentalobular = 16,
		Frearson = 17,
		SnakeEyes = 18,
		TA = 19,
		TP3 = 20,
		MorTorq = 21,
		ClutCHG = 22,
		ClutCHA = 23,
		GroupEyes = 24
	};

	/**
	* \brief class of the slits 
	*/
	class APICALL Slit {
	public:
		/**
		 * \brief constructor
		 */
		Slit();
		/**
		 * \brief slit constructor.
		 * \param w width
		 * \param h height
		 * \param d depth
		 * \param t type
		 */
		Slit(float w, float h, float d, SlitType t);
		/**
		 * \brief set the slit type.
		 * \param _t type value
		 * \return slit reference
		 */
		Slit& type(const int& _t);
		/**
		 * \brief get the slit type.
		 * \return type value
		 */
		int type() const;
		/**
		 * \brief set the width.
		 * \param _w width value
		 * \return slit reference
		 */
		Slit& width(const float& _w);
		/**
		 * \brief get the width.
		 * \return width value
		 */
		float width() const;
		/**
		 * \brief set the height.
		 * \param _h height value
		 * \return slit reference
		 */
		Slit& height(const float& _h);
		/**
		 * \brief get the height.
		 * \return height value
		 */
		float height() const;
		/**
		 * \brief set the depth.
		 * \param _d depth value
		 * \return slit reference
		 */
		Slit& depth(const float& _d);
		/**
		 * \brief get the depth.
		 * \return depth value
		 */
		float depth() const;

	private:
		float _width;
		float _height;
		float _depth;
		int _type;
	};

	/**
	 * \brief enumeration the types of the bolt head
	 */
	enum BoltHeadType {
		BoltNone = -1,	// none
		BoltHexa = 0, // NutHexaParams
		Countersunk = 1,// countersunk(flat)
		BoltRound = 2,	// round	
		Pan = 3,		// pan
		Dome = 4,		// dome
		Oval = 5,       // oval
		Square = 6,     // square
		TShaped = 7,    // t-shaped
		Cylinder = 8,   // cylinder
		Lamb = 9,       // LambParams
		Rim = 10,       // RimParams
		Eye = 11,		// EyeParams
		Bugle = 12,		// bugle
		Clop = 13		// clop  (self-tapping screw)
	};

	/**
		 * \brief class HeadParams
		 */
	class APICALL HeadParams {
	public:
		/**
		 * \brief constructor
		 */
		HeadParams();
		/**
		 * \brief constructor.
		 * \param _type head type
		 * \param _param pointer to the param data
		 */
		HeadParams(int _type, void* _param);
		/**
		 * \brief destructor
		 */
		~HeadParams();
		/**
		 * \brief set the parameters data with specified type.
		 * \param _type head type
		 * \param _param pointer to the param data
		 */
		HeadParams& setData(int _type, void* param);
		/**
		 * \brief get the head data
		 * \return pointer to the head data
		 */
		void* getData() const;
		/**
		 * \brief assignment operator
		 */
		HeadParams& operator = (const HeadParams& h);
		/**
		 * \brief copies the HeadParams object
		 */
		void copy(const HeadParams& h);
		/**
		 * \brief release the data
		 */
		void release();
	private:
		int _what;
		void* _data;
	};


	/**
		 * \brief HeadBaseParams struct of the head data
		 */
	class APICALL HeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		HeadBaseParams();
		/**
		 * \brief constructor.
		 * \param _d diameter
		 * \param _h height
		 */
		HeadBaseParams(float _d, float _h);
		/**
		 * \brief set the diameter.
		 * \param _d diameter
		 * \return HeadBaseParams reference
		 */
		HeadBaseParams& diameter(float _d);
		/**
		 * \brief set the height.
		 * \param _h height
		 * \return HeadBaseParams reference
		 */
		HeadBaseParams& height(float _h);
		/**
		 * \brief () operator
		 */
		operator void* () const;
		float _diameter;
		float _height;
	};

	/**
		 * \brief struct of the TShapedParams data
		 */
	class APICALL TShapedParams {
	public:
		/**
		 * \brief constructor.
		 */
		TShapedParams();
		/**
		 * \brief constructor.
		 * \param _d diameter
		 * \param _h height
		 * \param _w width
		 */
		TShapedParams(float _d, float _h, float _w);
		/**
		 * \brief set the diameter.
		 * \param _d diameter
		 * \return TShapedParams reference
		 */
		TShapedParams& diameter(float _d);
		/**
		 * \brief set the height.
		 * \param _h height
		 * \return TShapedParams reference
		 */
		TShapedParams& height(float _h);
		/**
		 * \brief set the width.
		 * \param _w width
		 * \return TShapedParams reference
		 */
		TShapedParams& width(float _w);
		/**
		 * \brief () operator
		 */
		operator void* () const;
		float _diameter;
		float _height;
		float _width;
	};

	/**
		 * \brief struct of the LambParams data
		 */
	class APICALL LambParams {
	public:
		/**
		 * \brief constructor.
		 */
		LambParams();
		/**
		 * \brief constructor.
		 * \param _l length
		 * \param _dtop top diameter
		 * \param _dbottom bottom diameter
		 * \param _hh head height
		 * \param _h height
		 * \param _thick thickness
		 */
		LambParams(float _l, float _dtop, float _dbottom, float _hh, float _h, float _thick);
		/**
		 * \brief set the length.
		 * \param _l length
		 * \return LambParams reference
		 */
		LambParams& length(float _l);
		/**
		 * \brief set the top diameter.
		 * \param _d top diameter
		 * \return LambParams reference
		 */
		LambParams& diameterTop(float _d);
		/**
		 * \brief set the bottom diameter.
		 * \param _d bottom diameter
		 * \return LambParams reference
		 */
		LambParams& diameterBottom(float _d);
		/**
		 * \brief set the height.
		 * \param _h height
		 * \return LambParams reference
		 */
		LambParams& height(float _h);
		/**
		 * \brief set the head height.
		 * \param _h height
		 * \return LambParams reference
		 */
		LambParams& headHeight(float _h);
		/**
		 * \brief set the thickness.
		 * \param _t thickness
		 * \return LambParams reference
		 */
		LambParams& thickness(float _t);
		/**
		 * \brief () operator
		 */
		operator void* () const;
		float _length;
		float _diameterTop;
		float _diameterBottom;
		float _headHeight;
		float _height;
		float _thickness;
	};

	/**
		 * \brief struct of the RimParams data
		 */
	class APICALL RimParams {
	public:
		/**
		 * \brief constructor.
		 */
		RimParams();
		/**
		 * \brief constructor.
		 * \param _d shoulder diameter
		 * \param _h shoulder height
		 * \param _dl inner ring diameter
		 * \param _d2 outer ring diameter
		 */
		RimParams(float _d, float _h, float _d1, float _d2);
		/**
		 * \brief set the shoulder diameter.
		 * \param _d diameter
		 * \return RimParams reference
		 */
		RimParams& shoulderDiameter(float _d);
		/**
		 * \brief set the shoulder height.
		 * \param _h height
		 * \return RimParams reference
		 */
		RimParams& shoulderHeight(float _h);
		/**
		 * \brief set the inner ring diameter.
		 * \param _d diameter
		 * \return RimParams reference
		 */
		RimParams& inRingDiameter(float _d);
		/**
		 * \brief set the outer ring diameter.
		 * \param _d diameter
		 * \return RimParams reference
		 */
		RimParams& outRingDiameter(float _d);

		/**
		 * \brief () operator
		 */
		operator void* () const;
		float _shoulderDiameter;
		float _shoulderHeight;
		float _inRingDiameter;
		float _outRingDiameter;
	};

	/**
		 * \brief struct of the EyeParams data
		 */
	class APICALL EyeParams {
	public:
		/**
		 * \brief constructor.
		 */
		EyeParams();
		/**
		 * \brief constructor.
		 * \param _thick thickness
		 * \param _dl inner ring diameter
		 * \param _d2 outer ring diameter
		 */
		EyeParams(float _thick, float _d1, float _d2);
		/**
		 * \brief set the inner ring diameter.
		 * \param _d diameter
		 * \return EyeParams reference
		 */
		EyeParams& inRingDiameter(float _d);
		/**
		 * \brief set the outer ring diameter.
		 * \param _d diameter
		 * \return EyeParams reference
		 */
		EyeParams& outRingDiameter(float _d);
		/**
		 * \brief set the thickness.
		 * \param _t thickness
		 * \return EyeParams reference
		 */
		EyeParams& thickness(float _t);
		/**
		 * \brief () operator
		 */
		operator void* () const;
		float _thickness;
		float _inRingDiameter;
		float _outRingDiameter;
	};
	/**
	* \brief bolt head primitive
	*/
	class APICALL boltHead : public prim {
	public:
		/**
		 * \brief constructor.
		 */
		boltHead();
		/**
		 * \brief set the head object.
		 * \param _h head obj reference
		 * \return bolt head reference
		 */
		boltHead& head(const HeadParams& _h);
		/**
		 * \brief get the HeadParams object.
		 * \return HeadParams object reference
		 */
		HeadParams& head();
		/**
		 * \brief get the const HeadParams object.
		 * \return HeadParams object reference
		 */
		const HeadParams& head() const;
		/**
		 * \brief set the head object with specified type and data.
		 * \param _type head type
		 * \param _data head data
		 * \return bolt head reference
		 */
		boltHead& head(const int& _type, void* data);
		/**
		 * \brief set the slit object.
		 * \param _s slit reference.
		 * \return bolt head reference
		 */
		boltHead& slit(const Slit& _s);
		/**
		 * \brief get the slit object.
		 * \return slit object reference
		 */
		 Slit& slit();
		 /**
		 * \brief get the const slit object.
		 * \return slit object reference
		 */
		 const Slit& slit() const;
		 /**
		  * \brief assignment operator
		  */
		 boltHead& operator = (const boltHead& h);
	protected:
		/**
		 * \brief creates a mesh container for the bolt head.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		HeadParams _head;
		Slit _slit;
	};

	/**
		 * \brief enumeration the types of the nut
		 */
	enum NutType {
		NutNone = -1,
		NutHexa = 0,
		Quard = 1,
		Acorn = 2,
		Lowacorn = 3,
		NutFlange = 4,
		Slits = 5,
		Radial = 6,
		NutLamb = 7,
		NutRim = 8,
		Selflock = 9,
		NutTShaped = 10,
		Clamplever = 11,
		NtCount = 12
	};


	class APICALL NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutHeadBaseParams();
		/**
		 * \brief constructor.
		 * \param t type of the nut
		 * \param d diameter
		 * \param h height
		 */
		NutHeadBaseParams(NutType t, float d, float h);
		/**
		 * \brief set the diameter.
		 * \param d diameter
		 * \return NutRadialParams reference
		 */
		NutHeadBaseParams& diameter(const float& d);
		/**
		 * \brief get the diameter.
		 * \return diameter
		 */
		float diameter() const;
		/**
		 * \brief set the height.
		 * \param h height value.
		 * \return NutHeadBaseParams reference
		 */
		NutHeadBaseParams& height(const float& h);
		/**
		 * \brief get the height.
		 * \return height value
		 */
		float height()const;
		/**
		 * \brief () operator
		 */
		operator void* () const;
		/**
		 * \brief set the nut type.
		 * \param t type value.
		 * \return NutHeadBaseParams reference
		 */
		NutHeadBaseParams& type(int t);
		/**
		 * \brief set the nut type.
		 * \return type value.
		 */
		int type() const;
		/**
		 * \brief copies the object.
		 * \param p pointer to the object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);

	private:
		int _type;
		float _diameter;
		float _height;
	};

	// NutHexaParams
	class APICALL NutHexaParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutHexaParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutHexaParams(float d, float h);
	};

	class APICALL NutQuardParams : public NutHeadBaseParams {
		/**
		 * \brief constructor.
		 */
		NutQuardParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutQuardParams(float d, float h);
	};

	class APICALL NutAcornParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutAcornParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 * \param h1 facet height
		 */
		NutAcornParams(float d, float h, float h1);
		/**
		 * \brief set the facet height.
		 * \param _h height value.
		 * \return NutAcornParams reference
		 */
		NutAcornParams& facetHeight(const float& _h);
		/**
		 * \brief get the facet height.
		 * \return facet height value.
		 */
		float facetHeight() const;
		/**
		 * \brief copies the NutAcornParams object.
		 * \param p pointer to the NutAcornParams object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an NutAcornParams object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);
	private:
		float _facetHeight;
	};

	class APICALL NutLowAcornParams : public NutAcornParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutLowAcornParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 * \param h1 facet height
		 */
		NutLowAcornParams(float d, float h, float h1);
	};

	class APICALL NutSelfLockParams : public NutAcornParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutSelfLockParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 * \param h1 facet height
		 */
		NutSelfLockParams(float d, float h, float h1);
	};

	class APICALL NutTShapedParams : public NutAcornParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutTShapedParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 * \param h1 facet height
		 */
		NutTShapedParams(float d, float h, float h1);
	};

	class APICALL NutFlangeParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutFlangeParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutFlangeParams(float d, float h);
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 * \param fw facet width
		 * \param fh facet height
		 */
		NutFlangeParams(float d, float h, float fw, float fh);
		/**
		 * \brief set the width.
		 * \param w width value.
		 * \return NutFlangeParams reference
		 */
		NutFlangeParams& facetWidth(const float& w);
		/**
		 * \brief set the facet height.
		 * \param h height value.
		 * \return NutFlangeParams reference
		 */
		NutFlangeParams& facetHeight(const float& h);
		/**
		 * \brief get the width.
		 * \return width value.
		 */
		float facetWidth() const;
		/**
		 * \brief get the height.
		 * \return height value.
		 */
		float facetHeight() const;
		/**
		 * \brief copies the NutFlangeParams object.
		 * \param p pointer to the NutFlangeParams object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an NutFlangeParams object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);
	private:
		float _facetWidth;
		float _facetHeight;
	};

	class APICALL NutRadialParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutRadialParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutRadialParams(float d, float h);
		/**
		 * \brief constructor
		 * \param d diameter
		 * \param h height.
		 * \param d1 hole diameter
		 * \param d2 hole depth
		 */
		NutRadialParams(float d, float h, float d1, float d2);
		/**
		 * \brief set the hole diameter.
		 * \param d diameter
		 * \return NutRadialParams reference
		 */
		NutRadialParams& holeDiameter(const float& d);
		/**
		 * \brief set the hole depth.
		 * \param d depth
		 * \return NutRadialParams reference
		 */
		NutRadialParams& holeDepth(const float& d);
		/**
		 * \brief set the hole diameter.
		 * \return hole diameter
		 */
		float holeDiameter() const;
		/**
		 * \brief get the hole depth.
		 * \return NutRadialParams hole depth
		 */
		float holeDepth() const;
		/**
		 * \brief set the hole place (0 - face, 1 - side).
		 * \param place place flag
		 * \return NutRadialParams reference
		 */
		NutRadialParams& holePlace(const int& place);
		/**
		 * \brief get the hole place.
		 * \return place flag 0 - face, 1 - side
		 */
		int holePlace() const;
		/**
		 * \brief copies the radial object.
		 * \param p pointer to the radial object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an radial object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);
	private:
		float _holeDiameter;
		float _holeDepth;
		int _holePlace;
	};

	class APICALL NutLambParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutLambParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutLambParams(float d, float h);
		/**
		 * \brief set the length.
		 * \param _l length
		 * \return NutLambParams reference
		 */
		NutLambParams& length(const float& _l);
		/**
		 * \brief set the bottom diameter.
		 * \param _d bottom diameter
		 * \return NutLambParams reference
		 */
		NutLambParams& diameterBottom(const float& _d);
		/**
		 * \brief set the top diameter.
		 * \param _d top diameter
		 * \return NutLambParams reference
		 */
		NutLambParams& diameterTop(const float& _d);
		/**
		 * \brief set the head height.
		 * \param _h height
		 * \return NutLambParams reference
		 */
		NutLambParams& headHeight(const float& _h);
		/**
		 * \brief set the thickness.
		 * \param _t thickness
		 * \return NutLambParams reference
		 */
		NutLambParams& thickness(const float& _t);
		/**
		 * \brief get the length.
		 * \return length value
		 */
		float length() const;
		/**
		 * \brief get the bottom diameter.
		 * \return bottom diameter
		 */
		float diameterBottom() const;
		/**
		 * \brief get the top diameter.
		 * \return top diameter
		 */
		float diameterTop() const;
		/**
		 * \brief get the head height.
		 * \param _h height
		 * \return height value
		 */
		float headHeight() const;
		/**
		 * \brief get the thickness.
		 * \return thickness value
		 */
		float thickness() const;
		/**
		 * \brief copies the NutLambParams object.
		 * \param p pointer to the NutLambParams object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an NutLambParams object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);

	private:
		float _diameterBottom;
		float _diameterTop;
		int _length;
		int _thickness;
		int _headHeight;
	};

	class APICALL NutSlitsParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutSlitsParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutSlitsParams(float d, float h);
		/**
		 * \brief set the width.
		 * \param w width value.
		 * \return NutSlitsParams reference
		 */
		NutSlitsParams& width(const float& w);
		/**
		 * \brief set the length.
		 * \param l length value.
		 * \return NutSlitsParams reference
		 */
		NutSlitsParams& length(const float& l);
		/**
		 * \brief set the count of NutSlitsParams.
		 * \param n count value.
		 * \return NutSlitsParams reference
		 */
		NutSlitsParams& count(const int& n);
		/**
		 * \brief get the width.
		 * \return width value.
		 */
		float width() const;
		/**
		 * \brief get the length.
		 * \return length value.
		 */
		float length() const;
		/**
		 * \brief get the count of NutSlitsParams.
		 * \return count value.
		 */
		int count() const;
		/**
		 * \brief copies the NutSlitsParams object.
		 * \param p pointer to the NutSlitsParams object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an NutSlitsParams object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);
	private:
		float _width;
		float _length;
		int _count;
	};

	class APICALL NutRimParams : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutRimParams();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutRimParams(float d, float h);
		/**
		 * \brief set the inner ring diameter.
		 * \param _d diameter
		 * \return NutRimParams reference
		 */
		NutRimParams& inRingDiameter(const float& _d);
		/**
		 * \brief set the outer ring diameter.
		 * \param _d diameter
		 * \return NutRimParams reference
		 */
		NutRimParams& outRingDiameter(const float& _d);
		/**
		 * \brief get the inner ring diameter.
		 * \return inner diameter
		 */
		float inRingDiameter() const;
		/**
		 * \brief get the outer ring diameter.
		 * \return outer diameter
		 */
		float outRingDiameter() const;
		/**
		 * \brief copies the NutRimParams object.
		 * \param p pointer to the NutRimParams object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an NutRimParams object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);
	private:
		float _inRingDiameter;
		float _outRingDiameter;
	};

	class APICALL NutClampLever : public NutHeadBaseParams {
	public:
		/**
		 * \brief constructor.
		 */
		NutClampLever();
		/**
		 * \brief constructor.
		 * \param d diameter
		 * \param h height
		 */
		NutClampLever(float d, float h);
		/**
		 * \brief set the diameter of the holder.
		 * \param _d diameter
		 * \return NutClampLever reference
		 */
		NutClampLever& holderDiameter(const float& _d);
		/**
		 * \brief set the length.
		 * \param _l length
		 * \return NutClampLever reference
		 */
		NutClampLever& length(const float& _l);
		/**
		 * \brief get the diameter of the holder.
		 * \return holder diameter
		 */
		float holderDiameter() const;
		/**
		 * \brief get the length.
		 * \return length value
		 */
		float length() const;
		/**
		 * \brief copies the NutClampLever object.
		 * \param p pointer to the NutClampLever object to copy. If the pointer equals to null then the object is duplicated
		 * \return the pointer to a copy of an NutClampLever object.
		 */
		virtual NutHeadBaseParams* copy(NutHeadBaseParams* p = nullptr);
	private:
		float _holderDiameter;
		float _length;
	};

	/**
	 * \brief nut primitive
	 */
	class APICALL nut : public prim {
	public:		
		/**
		 * \brief constructor.
		 */
		nut();
		/**
		 * \brief destructor.
		 */
		~nut();
		/**
		 * \brief set the typed data.
		 * \param data pointer to the data
		 * \return nut reference
		 */
		nut& setTypeData(void* data);
		/**
		 * \brief get the typed data.
		 * \return pointer to the data
		 */
		void* getTypeData();
		/**
		 * \brief set the hole thread diameter.
		 * \param _d diameter
		 * \return nut reference
		 */
		nut& threadDiameter(const float& d);
		/**
		 * \brief get the hole thread diameter.
		 * \return diameter
		 */
		float threadDiameter() const;
		/**
		 * \brief set the thread pitch.
		 * \param _p pitch
		 * \return nut reference
		 */
		nut& pitch(const float& p);
		/**
		 * \brief get the thread pitch.
		 * \return pitch
		 */
		float pitch() const;
		/**
		 * \brief set the enabled thread.
		 * \param _f flag enabled
		 * \return nut reference
		 */
		nut& enableThread(const bool& f);
		/**
		 * \brief get the enabled thread.
		 * \return enabled flag 
		 */
		bool enableThread() const;
		/**
		 * \brief set the nut thread profile.
		 * \param _t profile type value (triangle,trapeze,rectangular,round,persistent)
		 * \return nut reference
		 */
		nut& threadType(const int& _t);
		/**
		 * \brief get the nut thread profile.
		 * \return thread type value.
		 */
		int threadType() const;

	protected:
		/**
		 * \brief release the typed data
		 */
		void release();
		/**
		 * \brief creates a mesh container for the nut.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		float _threadDiameter;
		float _pitch;
		int _threadType;
		bool _enableThread;
		void* _typeData;
	};

	enum ThreadSurface {
		ThreadCylinder,
		ThreadCone,
		ThreadEdge
	};

	/**
	 * \brief bolt primitive
	 */
	class APICALL bolt : public prim {
	public:
		/**
		 * \brief constructor.
		 */
		bolt();
		/**
		 * \brief Holds the general information about bolt parameters.
		 */
		struct Info {
			float _diameter;	// thread diameter
			float _pitch;		// screw thread step
			float _theight;		// screw thread height
			float _length;		// bolt length
			float _tlength;		// screw thread length
			int   _ttype;		// screw thread profile type (triangle,trapeze,rectangular,round)
			int	  _uhead;		// underhead (-1-none, 0-cylinder,1-rect)
			float _uwidth;		// underhead width
			float _uheight;		// underhead height
			boltHead _head;		// head
			int   _nuttype;		// nut type
			float _nutloc;		// nut location reference thread
			float _nutHeight;   // nut height
		};
		/**
		 * \brief set the head object.
		 * \param _h head obj reference
		 * \return bolt reference
		 */
		bolt& head(const boltHead& _h);
		/**
		 *\brief get the head object.
		 * \return head obj reference
		 */
		const boltHead& head() const;
		/**
		 * \brief set the bolt diameter.
		 * \param _d diameter
		 * \return bolt reference
		 */
		bolt& diameter(const float& _d);
		/**
		 * \brief get the bolt diameter.
		 * \return diameter
		 */
		float diameter();		
		/**
		 * \brief set the thread pitch.
		 * \param _p pitch
		 * \return bolt reference
		 */
		bolt& pitch(const float& _p);
		/**
		 * \brief get the thread pitch.
		 * \return pitch
		 */
		float pitch();
		/**
		 * \brief set the thread height.
		 * \param _h height
		 * \return bolt reference
		 */
		bolt& threadHeight(const float& _h);
		/**
		 * \brief get the thread height.
		 * \return height value
		 */
		float threadHeight();
		/**
		 * \brief set the bolt length.
		 * \param _l length
		 * \return bolt reference
		 */
		bolt& length(const float& _l);
		/**
		 * \brief get the bolt length.
		 * \param length value
		 */
		float length();
		/**
		 * \brief set the thread length.
		 * \param _l length
		 * \return bolt reference
		 */
		bolt& threadLength(const float& _l);
		/**
		 * \brief get the thread length.
		 * \return length
		 */
		float threadLength();	
		/**
		 * \brief set the screw thread profile.
		 * \param _t profile type value (triangle,trapeze,rectangular,round,persistent)
		 * \return bolt reference
		 */
		bolt& threadType(const int& _t);
		/**
		 * \brief get the screw thread profile.
		 * \return thread type value.
		 */
		int threadType();
		/**
		 * \brief set the under head type.
		 * \param _uh type value (0-cylinder,1-rect)
		 * \return bolt reference
		 */
		bolt& underhead(const int& _uh);
		/**
		 * \brief get the under head type.
		 * \return under head type value
		 */
		int	underhead();
		/**
		 * \brief set the underhead width.
		 * \param _uw width
		 * \return bolt reference
		 */
		bolt& uwidth(const int& _uw);
		/**
		 * \brief get the underhead width.
		 * \return width value
		 */
		float uwidth();
		/**
		 * \brief set the underhead height.
		 * \param _uh height
		 * \return bolt reference
		 */
		bolt& uheight(const int& _uh);
		/**
		 * \brief get the underhead height.
		 * \return height value
		 */
		float uheight();
		/**
		 * \brief set the nut type.
		 * \param _t type value (see nut::Type enumarate)
		 * \return bolt reference
		 */
		bolt& nutType(const int& _t);
		/**
		 * \brief get the nut type.
		 * \return type value
		 */
		int nutType();
		/**
		 * \brief set the nut location on the bolt.
		 * \param _loc location value
		 * \return bolt reference
		 */
		bolt& nutLocation(const float& _loc);
		/**
		 * \brief get the nut location on the bolt.
		 * \return location value
		 */
		float nutLocation();
		/**
		 * \brief set the nut height on the bolt.
		 * \param _h height value
		 * \return bolt reference
		 */
		bolt& nutHeight(const float& _h);
		/**
		 * \brief get the nut height on the bolt.
		 * \return location value
		 */
		float nutHeight();

	protected:
		/**
		 * \brief creates a mesh container for the bolt.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		/**
		* \brief gets the information about nut.
		* \return nut parameters info
		*/
		Info& info();
		const Info& info() const;
		Info _info;
	};

	/**
	 * \brief screw primitive
	 */
	class APICALL screw : public prim {
	public:
		/**
		 * \brief Holds the general information about bolt parameters.
		 */
		struct Info {
			float _diameter;	// core diameter
			float _pitch;		// screw thread step
			float _theight;		// screw thread height
			float _tlength;		// screw thread length
			float _tdiameter;	// screw thread diameter
			float _length;		// screw length
			int	  _uhead;		// underhead (-1 none, 0-cylinder,1-rect)
			int   _uwidth;		// underhead width
			int   _uheight;		// underhead height
			boltHead _head;		// bolt head
		};
		/**
		 * \brief constructor.
		 */
		screw();
		/**
		 * \brief set the bolt head object.
		 * \param _h bolt head obj reference
		 * \return screw reference
		 */
		screw& head(const boltHead& _h);
		/**
		 *\brief get the bolt head object.
		 * \return bolt head obj reference
		 */
		const boltHead& head() const;
		/**
		 * \brief set the screw diameter.
		 * \param _d diameter
		 * \return screw reference
		 */
		screw& diameter(const float& _d);
		/**
		 * \brief get the screw diameter.
		 * \return diameter
		 */
		float diameter();
		/**
		 * \brief set the screw thread step(pitch).
		 * \param _p step
		 * \return screw reference
		 */
		screw& pitch(const float& _p);
		/**
		 * \brief get the screw thread step(pitch).
		 * \return step value
		 */
		float pitch();
		/**
		 * \brief set the thread diameter.
		 * \param _d diameter
		 * \return screw reference
		 */
		screw& threadDiameter(const float& _d);
		/**
		 * \brief get the thread diameter.
		 * \return diameter
		 */
		float threadDiameter();
		/**
		 * \brief set the screw thread height.
		 * \param _h height
		 * \return screw reference
		 */
		screw& threadHeight(const float& _h);
		/**
		 * \brief get the screw thread height.
		 * \return thread height value
		 */
		float threadHeight();
		/**
		 * \brief set the screw thread length.
		 * \param _l length
		 * \return screw reference
		 */
		screw& threadLength(const float& _l);
		/**
		 * \brief get the screw thread length.
		 * \return length value
		 */
		float threadLength();
		/**
		 * \brief set the screw length.
		 * \param _l length
		 * \return screw reference
		 */	
		screw& length(const float& _l);
		/**
		 * \brief get the screw length.
		 * \return length
		 */
		float length();
		/**
		 * \brief set the underhead type.
		 * \param _uh type (0-cylinder,1-rect)
		 * \return screw reference
		 */
		screw& underhead(const int& _uh);
		/**
		 * \brief get the underhead type.
		 * \return type value
		 */ 	
		int	underhead();
		/**
		 * \brief set the underhead width.
		 * \param _uw width
		 * \return screw reference
		 */
		screw& uwidth(const int& _uw);
		/**
		 * \brief get the underhead width.
		 * \return width value
		 */
		float uwidth();
		/**
		 * \brief set the underhead height.
		 * \param _uh height
		 * \return screw reference
		 */
		screw& uheight(const int& _uh);
		/**
		 * \brief get the underhead height.
		 * \return height value
		 */
		float uheight();
	protected:
		/**
		 * \brief creates a mesh container for the screw.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		/**
		* \brief gets the information about screw.
		* \return screw parameters info
		*/
		Info& info();
		const Info& info() const;
		Info _info;
	};

	/**
	 * \brief washer primitive
	 */
	class APICALL washer : public prim {
	public:
		/**
		 * \brief enumeration the types of the washer
		 */
		enum Type {
			None = -1,
			Flat = 0,
			Grover = 1,
			Plate = 2,
			Wave = 3,
			Quard = 4,
			Sphere = 5,
			Conus = 6,
			Springlock = 7
		};
		/**
		 * \brief constructor.
		 */
		washer();
		/**
		 * \brief set the inner diameter.
		 * \param _d diameter
		 * \return washer reference
		 */
		washer& innerDiameter(const float& _d);
		/**
		 * \brief get the inner diameter.
		 * \return diameter
		 */
		float innerDiameter() const;
		/**
		 * \brief set the outer diameter.
		 * \param _d diameter
		 * \return washer reference
		 */
		washer& outerDiameter(const float& _d);
		/**
		 * \brief get the outer diameter.
		 * \return  diameter
		 */
		float outerDiameter() const;
		/**
		 * \brief set the diameter of the conus washer.
		 * \param _d diameter
		 * \return washer reference
		 */
		washer& conusDiameter(const float& _d);
		/**
		 * \brief get the diameter of the conus washer.
		 * \return diameter
		 */
		float conusDiameter() const;
		/**
		 * \brief set the washer thickness.
		 * \param _s thickness value
		 * \return washer reference
		 */
		washer& thickness(const float& _s);
		/**
		 * \brief get the washer thickness.
		 * \return thickness value
		 */
		float thickness() const;
		/**
		 * \brief set the washer height.
		 * \param _h height value
		 * \return washer reference
		 */
		washer& height(const float& _h);
		/**
		 * \brief get the washer height.
		 * \return height value
		 */
		float height() const;
		/**
		 * \brief indicates the facet usage.
		 * \param _f facet flag
		 * \return washer reference
		 */
		washer& facet(const bool& _f);
		/**
		 * \brief get the facet flag.
		 * \return facet flag value
		*/
		bool facet() const;
		/**
		 * \brief set the washer type.
		 * \param _t type value
		 * \return washer reference
		 */
		washer& type(washer::Type _t);
		/**
		 * \brief get the washer type.
		 * \return type value
		 */
		washer::Type type() const;

	protected:
		/**
		 * \brief creates a mesh container for the washer.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		float _innerDiameter;
		float _outerDiameter;
		float _conusDiameter;
		float _thickness;
		float _height;
		bool _facet;
		Type _type;
	};

	/**
	 * \brief freeform primitive
	 */
	class APICALL freeform: public prim {
	public:
		typedef	list<vec3> Points3D;
		struct ffElement {
			str name;
			list<str> subpaths;
		};
		typedef list<ffElement> ffObjsList;
		struct ffInfo {
			str name;
			str subName;
			bool symX;
			bool symY;
			bool symZ;
			Points3D points;
		};
		/**
		* \brief constructor
		*/
		freeform();
		/**
		* \brief constructor
		* \param _ffname free form name
		* \param _ffsubname free form subname
		*/
		freeform(const char *_ffname, const char* _ffsubname);
		/**
		 * \brief destructor
		 */
		~freeform();
		/**
		 * \brief Enable the XYZ-mirror symmetry
		 * \param x true to enable x-symmetry, false to disable
		 * \return freeform reference
		 */	
		 freeform& symx(const bool& x);
		/**
		 * \brief Enable the XYZ-mirror symmetry
		 * \param y true to enable y-symmetry, false to disable
		 * \return freeform reference
		 */
		freeform& symy(const bool& y);
		/**
		 * \brief Enable the XYZ-mirror symmetry
		 * \param z true to enable z-symmetry, false to disable
		 * \return freeform reference
		 */
		freeform& symz(const bool& z);
		/**
		 * \brief set the free form size  
		 * \param x size in the x - axis
		 * \param y size in the y - axis
		 * \param z size in the z - axis
		 * \return freeform reference
		 */
		freeform& size(float x, float y, float z);
		/**
		 * \brief set the free form size
		 * \param v vector size
		 * \return freeform reference
		 */
		freeform& size(const vec3& v);
		/**
		 * \brief set the free form name.
		 * \details set of names {"Cube","Patch,"Blob","Round","Ring",Torus","Tube2","Tube3","Disc","Cylinder"}
		 * \param name the name string
		 * \return the freeform reference
		 */
		freeform& ffname(const char* name);
		/**
		 * \brief get the free form name.
		 * \return the name
		 */
		const char* ffname() const;
		/**
		 * \brief set the free form sub name.
		 * \param name the name string
		 * \return the freeform reference
		 */
		freeform& ffsubname(const char* name);
		/**
		 * \brief get the free form sub name.
		 * \return the name
		 */
		const char* ffsubname() const;
		/**
		 * \brief set the knot point of the primitive.
		 * \param i point index
		 * \param point the coordinates of the point
		 * \return the freeform reference
		 */
		freeform& SetPoint(int i, const vec3& point);
		/**
		 * \brief get the account of the knot points
		 * \return count of points
		 */
		int CountPoints() const;
		/**
		 * \brief reset the knot points
		 */
		void ResetPoints();
		/**
		 * \brief get the sub names for current free form name.
		 * \param ffName free form name
		 * \return the pointer to list string of sub names
		 */
		list<str>* ffSubNamesList();
		/**
		 * \brief get the sub names for specified free form name.
		 * \param ffName free form name
		 * \return the pointer to list string of sub names
		 */
		list<str>* ffSubNamesList(const str& ffName);
		/**
		* \brief gets the object's list.
		* \return objs list reference
		*/
		ffObjsList& objsList();
		/**
		 * \brief get the knot(control) points of the primitive.
		 * \return the points reference
		 */
		Points3D& ffControlPoints();
	protected:
		/**
		 * \brief creates a mesh container for the screw.
		 * \return mesh pointer
		 */
		virtual cMeshContainer* create_mesh();
	private:
		/**
		 * \brief set the default points of the primitive.
		 * \return the freeform reference
		 */
		void SetupPoints();
		/**
		 * \brief set the objects paths.
		 * \return the freeform reference
		 */
		void SetObjectPaths();
		/**
		* \brief gets the information about screw.
		* \return screw parameters info
		*/
		ffInfo& info();
		const ffInfo& info() const;
		/*
		 * \brief gets the current object name.
		 * \return object name 
		 */
		str& objName();

		ffInfo _info;
		ffObjsList _ffobjs;
		list<str>* _subNames;
		str _objName;
		vec3 _side;
		int _ffDim[3];
	};

}
#pragma pack(pop)
