#pragma once

/**
 * \brief This class keeps tree-like structure that consists of pair TAG : VALUE + LIST of TagList-s\n
 * It is equivalent to XML without attributes. This is very fast and lightweight version of XML/JSON/PB\n
 * The structure may be stored in binary(compact) form. It may contain binary data, not only the text.
 * It is very good for saving data into the scene using VoxelExtension::StoreData/RestoreData.
 * There are a lot of examples of usage in VoxelExtension.h/cpp\n
 * This structure extensively used for the BaseClass serialization. 
 * 
 * 
 */
class BinStream;
class BaseClass;

/**
 * \brief This class intended to load/save the class derived from the BaseClass as XML/JSON. But it may be used just for
 * XML/JSON load/save. Pay attention, attributes of the XML not supported.
 */
class APICALL TagsList
{
public:
	TagsList();
	~TagsList();
	TagsList(const char* QuoteName);
	//new/delete are using pool, so this is fast
	void operator delete(void* ptr, size_t size);
	void* operator new(size_t size);

	///convert tag value to int
	int to_int () const;
	///convert tag value to float
	float to_float() const;
	///convert tag value to bool
	bool to_bool() const;
	///get value of the tag
	const char* to_string() const;
	///get value of the tag as float
	float to_float();
	///get value of the tag as the data array, returns the actual data length
	int get_data(void* buffer, int maxlength);
	
	///last saved/loaded file with tags
	static comms::cStr& LastTagFile();
	static comms::cStr& LastTag();
	static comms::cStr& defTab();

	///convert string to the tags structure
	int GetTagsFromString(const char* XMLSource);
	///convert tags structure to the string
	int PutTagsIntoString(comms::cStr* To, int shift = 0, bool fast = false);
	///convert string to the tags structure
	int GetTagsFromJsonString(const char* jsonSource);
	///convert tags structure to the string
	int PutTagsIntoJsonString(comms::cStr* Source, int shift = 0, bool treatAllAsText = false);
	///read tags from file
	bool ReadTagsFromFile(const char* FilePath);
	///save tags to file
	void WriteTagsToFile(const char* FilePath, bool OnlyChanged = false);

	///assign int value to the tag value
	void Assign(int v);
	///assign string value to the tag
	void Assign(const char* s);
	// get/set parent for this tag 
	void SetParent(TagsList* Parent);
	TagsList* GetParent() const;

	///Add already allocated tag to the tags list within this tag
	TagsList* AddSubTag(TagsList* SubQ);
	///Add new tag to the tags list within this tag
	TagsList* AddSubTag(const char* quotename, const char* body = "");
	///Add new tag with integer value  to the tags list within this tag
	TagsList* AddSubTag(const char* quotename, int body);
	TagsList* AddSubTag(const char* quotename, DWORD body);
	TagsList* AddSubTag(const char* quotename, short body);
	TagsList* AddSubTag(const char* quotename, WORD body);
	TagsList* AddSubTag(const char* quotename, char body);
	///Add new tag with the floating point value
	TagsList* AddSubTag(const char* quotename, float body);
	///Add new tag with the data array
	TagsList* AddSubTag(const char* quotename, void* data, int Length);
	///add tags using << operator. Add tag name (should start with # sign), then pass the value. Tags names will be treated without the # sign.
	///Example: tags << "#x" << x << "#name" << name;
	///You may pass values without tags, then tags names will be treated as empty
	TagsList& operator << (const char* string);
	TagsList& operator << (int value);
	TagsList& operator << (bool value);
	TagsList& operator << (float value);
	TagsList& operator << (const comms::cVec2& value);
	TagsList& operator << (const comms::cVec3& value);
	TagsList& operator << (const comms::cVec4& value);
	TagsList& operator << (const comms::cMat3& value);
	TagsList& operator << (const comms::cMat4& value);
	TagsList& operator << (BaseClass* BC);
	///store the list of simple elements. Elements should consist of simple types. Pointers, virtual functions or nontrivial constructors strictly prohibited there!
	///This is intended to keep simple arrays of values, vectors etc.
	template <class X>
	TagsList& operator << (const cList<X>& array);
	/**This template allows to store the given field of the array into the TagsList
	* It is helpful if you need to store separate fields of some complex structure thta may not be serialized as a whole
	* Example: you need to store Vector3D::x into the list -
	* \code
	* cList<Vector3D> v3list;
	* ....add elements...
	* FagsList T;
	* T.StoreField("V3X",v3list,&Vector3D::x);//store
	* ....
	* cList<Vector3D> v3list1;
	* T.elem("V3X").RestoreField(v3list1,&Vector3D::x);//now v3list1 has filled x field
	* \endcode
	*/
	template <typename X, typename TYPE>
	void StoreField(const char* tagname, const cList<X>& array, TYPE X::* member);

	///read values from the tags list
	void operator >> (comms::cStr& string_value);
	void operator >> (int& value);
	void operator >> (float& value);
	void operator >> (bool& value);
	void operator >> (comms::cVec2& value);
	void operator >> (comms::cVec3& value);
	void operator >> (comms::cVec4& value);
	void operator >> (comms::cMat3& value);
	void operator >> (comms::cMat4& value);
	void operator >> (BaseClass* BC);
	///restore simple list, the size of the element should be the same as ou stored
	template <class X>
	void operator >> (cList<X>& array);
	///This template is complimentary to the StoreField. It allows to restore previously stored field of the structure.
	///See the StoreField description.
	template <typename X, typename TYPE>
	void RestoreField(cList<X>& array, TYPE X::* member);
	
	
	///save to the binary stream in the binary form, see the implementation for the data format specification
	void bin_ToBS(BinStream& BS);
	///save to the binary stream in the textual (XML) form
	void text_ToBS(BinStream& BS);
	///load in binary or text form
	bool FromBS(BinStream& BS);
	
	comms::cStr& body();
		
	///Get sub-tag by index
	TagsList* GetSubTag(int Index);
	///returns the reference to the element. Pay attention, it may return the zero reference!
	///Anyway, it may be useful for operations like T->elem(i) >> x. It works even it the element i-th does not exist, in this case x will be just cleared.
	TagsList* elem(int Index); /// Should return pointer (not reference) because of optimization error in clang/LLVM
	TagsList* operator[](int Index);
	///Get sub-tag by name
	TagsList* GetSubTag(const char* SubQuoteName);
	int GetSubTagIndex(const char* SubQuoteName);
	///returns the reference to the element. Pay attention, it may return the zero reference!
	///Anyway, it may be useful for operations like T->elem("X") >> x. It works even it the element "X" does not exist, in this case x will be just cleared.
	TagsList* elem(const char* SubQuoteName); /// Should return pointer (not reference) because of optimization error in clang/LLVM
	TagsList* operator[](const char* SubQuoteName);
	///Get amount of sub-tags
	int GetSubTagsCount();
	///Get amount of sub-tags, same as previous, js-like
	int length();

	///Remove sub-tag by index
	void RemoveSubTag(int idx);
	///Remove sub-tag by index
	void RemoveSubTag(const char* id);
	///Remove and return tag by index. The return value will not be destroyed, you should do it later by yourself.
	TagsList* RemoveAndGetSubTag(int idx);

	///returns this tag name
	const char* GetTagName();
	///set name for the current tag. If this tag was already used somewhere, the memory will not be allocated for the name.
	void SetTagName(const char* Name);
	///free all memory associated with this tag
	void FreeTag();
	///copy 
	void operator = (TagsList& src);
	///This function returns constant location for the input string. Even if the input string will be destroyed later, the return value will be kept till the program termination.
	static const char* GetConstantLocation(const char* str);
	static int& GetConstantLocationValueRef(const char* str);
	///This function returns constant location for the input sub - string (max length = Len). Even if the input string will be destroyed later, the return value will be kept till the program termination.
	static const char* GetConstantSubstringLocation(const char* str, int Len, int** value = nullptr);

	void setAsArray(bool isArray);
	bool isArray() const;
private:
	const char* _tagname;
	const char* _tag(const char* tag);
    comms::cList<TagsList*> SubTags;
	comms::cStr TagBody;
	TagsList* _parent;
	bool _isArray;

};

template <class X>
void TagsList::operator>>(cList<X>& array) {
	array.Free();
	if (this) {
		const char* s = body().ToCharPtr();
		if (body().Length()) {
			cPtrDiff num = *((int*)s);
			cPtrDiff sz = *((int*)(s + 4));
			if (sz == sizeof(X)) {
				X val;
				array.Add(val, num);
				memcpy(array.ToPtr(), s + 8, num * sz);
			}
		}
	}
}

template <typename X, typename TYPE>
void TagsList::RestoreField(cList<X>& array, TYPE X::* member) {
	if (this) {
		const char* s = body().ToCharPtr();
		if (body().Length()) {
			cPtrDiff num = *((int*)s);
			cPtrDiff sz = *((int*)(s + 4));
			if (sz == sizeof(TYPE)) {
				TYPE val;
				if (array.Count() < num) {
					X ref;
					array.Add(ref, num - array.Count());
				}
				TYPE* TL = (TYPE*)(s + 8);
				for (int i = 0; i < num; i++) {
					array[i].*member = *TL;
					TL++;
				}
			}
		}
	}
}

template <class X>
TagsList& TagsList::operator<<(const cList<X>& array) {
	TagsList* T = AddSubTag(LastTag());
	if (array.Count()) {
		T->body().SetLength(4 + 4 + array.Count() * sizeof(X));
		char* s = T->body().ToNonConstCharPtr();
		*((int*)s) = array.Count();
		*((int*)(s + 4)) = sizeof(X);
		memcpy(s + 8, array.ToPtr(), cPtrDiff(array.Count()) * sizeof(X));
	}
	LastTag().Clear();
	return *T;
}

template <typename X, typename TYPE>
void TagsList::StoreField(const char* tagname, const cList<X>& array, TYPE X::* member) {
	TagsList* T = AddSubTag(tagname);
	if (array.Count()) {
		T->body().SetLength(4 + 4 + array.Count() * sizeof(TYPE));
		char* s = T->body().ToNonConstCharPtr();
		*((int*)s) = array.Count();
		*((int*)(s + 4)) = sizeof(TYPE);
		TYPE* TL = (TYPE*)(s + 8);
		for (int i = 0; i < array.Count();i++) {
			*TL = array[i].*member;
			TL++;
		}
	}
}

void ToBase64(BYTE* buf, int Len, comms::cStr& dest);
void ToBase64(BYTE* buf, int Len, std::string& dest);
int ReadBase64(const char* src, BYTE* buf, int MaxLen);

TagsList* new_TagsList();
inline TagsList* new_TagsList(const char* s){
	TagsList* x=new_TagsList();
	x->SetTagName(s);
	return x;
}
void delete_TagsList(TagsList* x);
inline void TagsList::operator delete(void* ptr, size_t size) {
	delete_TagsList((TagsList*)ptr);
}
inline void* TagsList::operator new(size_t size) {
	return new_TagsList();
}

inline comms::cStr& TagsList::body() {
	return TagBody;
}

inline void TagsList::setAsArray(bool isArray) {
	this->_isArray = isArray;
}

inline bool TagsList::isArray() const {
	return _isArray;
}


