// The basic generator example, just to demonstrate the principles of generators, look the \ref TreesGenerator.cpp for the more sophisticated example. 
#include <CoreAPI.h>
//@config: Release

using namespace coat;

// This is the simplest non-destructive generator example.
// It is registered as tool in the current room toolset as the first tool in list.

const char* file_for_settings = "data/Temp/SimpleGenerator.json";

class SimpleGenerator : public SculptGenerator{
	coat::Mesh m;
public:
	SimpleGenerator() {
		numSpheres = 200;
		FigureRadius = 30;
		FigureRadiusVariation = 0.1;
		SpheresRadius = 10;
		SpheresRadiusVariation = 0.5;
		// load the sphere mesh
		m.Read("data/Samples/Sphere.obj");
	}
	// returns the name/ID of the tool in the toolset
	virtual const char* GetID() {
		return "SimpleGenerator";
	}

	virtual const char* getDefaultObjectName() {
		return "Spheres";
	}

	int numSpheres;
	float FigureRadius;
	float FigureRadiusVariation;
	float SpheresRadius;
	float SpheresRadiusVariation;

	SERIALIZE() {
		FUNCTION_CALL(CreateNewObject,"CreateNewObject");
		// we show parameters only if the object already created
		if (getObject()) {
			SLIDER(numSpheres, "numSpheres", 1, 1000);
			FSLIDER(FigureRadius, "FigureRadius", 1, 100, 1, false);
			FSLIDER(FigureRadiusVariation, "FigureRadiusVariation", 0, 1, 100, false);
			FSLIDER(SpheresRadius, "SpheresRadius", 1, 100, 1, false);
			FSLIDER(SpheresRadiusVariation, "SpheresRadiusVariation", 0, 1, 100, false);

			// show the transform gizmo, both TransformObject and Transform defined in the parent class
			REG_AUTO(TransformObject);
			if(TransformObject) {
				NOHASH{
					REG_AUTO(Transform);
				}
			}
		}
	}

	// generate random value with variation
	float random(float value, float variation) {
		return value * (1.0f + variation * rand() / 32768.0f);
	}

	void generate(Volume v) {
		coat::Mesh summ;
		for (int i = 0; i < numSpheres; i++) {
			// create transformation matrix.
			// scaling to the sphere radius
			coat::mat4 transform = coat::mat4::Scaling(random(SpheresRadius, SpheresRadiusVariation));
			// translate using the random vector
			transform *= coat::mat4::Translation(coat::vec3::RandNormal() * random(FigureRadius, FigureRadiusVariation));
			// add the sphere to the summary mesh
			summ.addTransformed(m, transform);
		}
		v.mergeMesh(summ, mat4::Identity, coat::BOOL_MERGE);
		// save settings
		WriteToFile(file_for_settings);
	}

	// fast ang rough generating, the function is overriden from the SculptGenerator
	virtual void GeneratePreview() override {
		SceneElement s(getObject());
		// we remove previously generates object
		s.removeSubtree();
		// we add the new object as sub-object to the current 
		SceneElement t = s.addChild("Simple");
		Volume v = t.Volume();
		// turen the new volume to surface
		v.toSurface();
		t.setTransform(mat4::Identity);
		// we generate as if there is no transform
		generate(t.Volume());
		// we apply the transform that user made manually 
		t.setTransform(t.getTransform() * s.getTransform());
		// we select the generated object
		s.selectOne();
	};
	// generage the object in "final quality", the function is overriden from the SculptGenerator
	
	virtual void GenerateFinalObject() override {
		// in simplest case the final quality object is same as preview
		GeneratePreview();
	}
};

EXPORT_EXTENSION(SimpleGenerator) {
	// we create the generator object
	SimpleGenerator* tg = new SimpleGenerator;
	// read settings
	tg->ReadFromFile(file_for_settings);
	// register the tool in the toolset
	VoxelExtension::Register(tg);
	ui::toRoom("Sculpt");
	// activate the tool
	tg->Activate();
}
